/*------------------------------------------------------------------------------*
 * File Name: stats_utils.h 													*
 * Creation:																	*
 * Purpose: OriginC header file	containing LLOCs for statistics tools.			*
 * Copyright (c) OriginLab Corp.	2003, 2004, 2005, 2006, 2007, 2008, 2009	*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 * Skip 04/09/04 CHANGE_BOOL_TO_INT                                             *
 * Fisher 12/08/08 QA80-12735  SUPPORT_USER_SUPPLIED_BIN_VECTOR_IN_FREQCOUNTS	*  
 *------------------------------------------------------------------------------*/

#ifndef _STATS_UTILS_H
#define _STATS_UTILS_H

#ifndef _OC_MATH_H
#include <ocmath.h>
#endif


#include <ocStatEx.h>
 
#include <stats_types.h>

#include <ReportTree.h>

////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
// Define constants, enums, typedefs, and non-localized strings
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

#define SU_MIN_NPTS					2
#define SU_DEFAULT_CONF_LEVEL		0.95
#define COX_MODEL_TOLERANCE			5e-6
#define COX_MODEL_ITERATIONS		100
#define REPORT_TABLE   100
///Jasmine 05/28/08 QA80-11234 v8.0872 GRID_XY_AFTER_GENERATE_XY_FOR_XYZ_FIT
#define XY_FIT_DEFAULT_POINTS		100
///Sophy 2/11/2009 v8.0976c CHANGE_DEFAULT_NUM_POINTS_FOR_XYZ_FITT_AS_CP_SUGGEST
//#define XYZ_FIT_DEFAULT_POINTS		10
#define XYZ_FIT_DEFAULT_POINTS		50
///end CHANGE_DEFAULT_NUM_POINTS_FOR_XYZ_FITT_AS_CP_SUGGEST
///End GRID_XY_AFTER_GENERATE_XY_FOR_XYZ_FIT

///Jasmine 05/30/08 QA80-11630 ADD_OPTION_FOR_SHOW_MARKER_IN_SRC_PLOT_AFTER_FIT
#ifdef	MARKER_VISIBILITY_OPTIONS
enum{
	SHOW_MARKER_ALWAYS 				= 0,
	SHOW_MARKER_IF_PARTIAL_RANGE,
	SHOW_MARKER_NEVER,
};
#endif //	MARKER_VISIBILITY_OPTIONS
///End ADD_OPTION_FOR_SHOW_MARKER_IN_SRC_PLOT_AFTER_FIT

///--- Iris 10/31/04 use to show error info in result table, CPY 11/2/04, moved here from stats_types.h
typedef struct tagErrInfo
{
	string 		strErrInfo;
}ErrInfo;
//---
// 2 Sample t-Test enum
enum
{
	EQUAL_VARIANCE = 0,
	UNEQUAL_VARIANCE,
};

////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
// Declare function prototypes
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////




/////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//				Basic Functions: Descriptive Statistics, Percentile, Range, Quantile, Frenqucy Count									
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute descriptive statistics (moments) with weighting passing results
		in a structure. Function ocmath_desc_stats is the main computational engine.
	Example:
		void test()
		{
			int nErr;
			vector vData1 = {7,1,11,11,7,11,3,1,2,21,1,11,10};
			vector pvweight1 = {0.3, 0.8, 0.9, 0.6, 0.15, 0.36, 0.7, 0.2, 0.9, 0.54, 0.1, 0.3, 0.2};
			double dConfLevel =0.9;
			DescStatResults dsr;
			nErr = stats_descriptive(vData1, dsr, &pvweight1, dConfLevel);

		}
	Parameters:
		vData=Input data on which descriptive statistics are computed
		dsr=Output results of descriptive statistics
		pvWeight=Optional input weighting vector, NULL if weighting not used
		dConfLevel=Optional input confidence level, default value is SU_DEFAULT_CONF_LEVEL
		iDenFlag = Optional input value indicates the variance divisor.default value is DS_SAS1_DOF
						DS_SAS1_DOF(default): Den = dof, which is also the default algorithm in SAS
						DS_SAS2_N: Den = n;
						DS_SAS3_WDF: Den = Sum(weight) - 1; Skewness and Kurtosis are set to missing in this case
						DS_SAS4_WGT: Den = Sum(weight); Skewness and Kurtosis are set to missing in this case
						DS_NAG: Den = Sum(weight^2) -sum(weight), which is the algorithm of nag. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_descriptive(const vector& vData, DescStatResults& dsr,
	 const vector* pvWeight = NULL, double dConfLevel = SU_DEFAULT_CONF_LEVEL, int iDenFlag = DS_NAG); // dConfLevel = SU_DEFAULT_CONF_LEVEL, pvWeight = NULL

/**$
		Function to compute descriptive statistics (moments) with weighting passing results
		in a structure. Function ocmath_basic_summary_stats and ocmath_opt_summary_stats is the main computational engine.
	Parameters:
		vData=Input data on which descriptive statistics are computed
		opt=Input structure containing descriptive statistics
		dsr=Output results of descriptive statistics
		pvWeight=Optional input weighting vector, NULL if weighting not used
		dConfLevel=Optional input confidence level, default value is SU_DEFAULT_CONF_LEVEL
		bmultimodes=Optional output indicator to indicate if there is multiple modes, NULL if Mode is not calculated
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_desc_stats(const vector& vData, const DescStatOptions& opt, DescStatResults& dsr,
	 const vector* pvWeight = NULL, double dConfLevel = SU_DEFAULT_CONF_LEVEL, bool* pmultimodes = NULL); // dConfLevel = SU_DEFAULT_CONF_LEVEL, pvWeight = NULL, pmultimodes = NULL

	 
// Comment last updated by Joseph on 09/11/2007
/** >Statistics>Advanced
		Function to compute quantiles including Min, iMin, Max, iMax, Range and IQR for a vector.
	Example:
		
		#include <stats_types.h>
		#include <stats_utils.h>

		void test()
		{
			vector vData = { 1.1, 0.1, 2.2, 4.3, -4.2, 5.2, 1.6, 7.2, 0.8, 3.5 };
			QuantileOptions opt;
			QuantileResults res;
			vector vPercents;
			vector vPercentiles;
			int nRet;
			opt.Min = true;
			opt.iMin = true;
			opt.Q1 = true;
			opt.Median = true;
			opt.Q3 = true;
			opt.Max = true;
			opt.iMax = true;
			opt.IQR = true;
			opt.Range = true;	
			vPercents.SetSize(4);
			vPercents[0] = 78;
			vPercents[1] = 41;
			vPercents[2] = 21;
			vPercents[3] = 6;	
			opt.Interpolate = INTERPOLATE_WEIGHT_AVER_RIGHT;	
			nRet = stats_quantiles( vData, opt, res, vPercents, vPercentiles );	
		}	
	Parameters:
		vData=Input vector containing data on which quantiles are computed
		opt=Input structure containing quantile options
		res=Output structure containing quantile results
		vPercents=Optional input vector containing custom percents for which percentiles are computed
		vPercentiles=Optional output vector containing custom percentiles (results)
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_quantiles(const vector& vData, const QuantileOptions& opt, QuantileResults& res,
	 const vector& vPercents = NULL, vector& vPercentiles = NULL, const vector* pvWeight = NULL);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute percentiles for a vector passing percents and percentiles in vectors.
	Example:
		void test()
		{
			vector vData(50);
			vData.Data(1,50,1);	
			vector vPercents = {100,99,95,90,75,50,25,10,5,1,0};
			int nSize = vPercents.GetSize();
			vector vResult;
			vResult.SetSize(nSize);	
			int 	i;
			int 	nRet;
			int 	nInterpolate;	
			nInterpolate = 1;
			vResult.RemoveAll();
			nSize = vPercents.GetSize();
			vResult.SetSize(nSize);
			nRet = stats_percentiles(vData, vPercents, vResult, nInterpolate);
		}	
	Parameters:
		vData=Input vector containing data on which percentiles are computed
		vPercents=Input vector containing percents for which percentiles are computed
		vPercentiles=Output vector containing percentiles (results)
		nInterpolate=Optional input indicating interpolation type, possible values include:
						INTERPOLATE_WEIGHT_AVER_LEFT 		Weighted average aimed at x(n + 1) p
						INTERPOLATE_WEIGHT_AVER_RIGHT	 	Weighted average aimed at xnp
						INTERPOLATE_NEAREST_NEIGHBOR		Observation numbered closest to np
						INTERPOLATE_EDF 					Empirical distribution function No interpolation(Default)
						INTERPOLATE_EDF_AVER				Empirical distribution function with averaging
		pvWeight=Optional input weighting vector, NULL if weighting not used.
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_percentiles(const vector& vData, const vector& vPercents, vector& vPercentiles,
	 int nInterpolate = 0, const vector* pvWeight = NULL); // Function to compute percentiles.

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute range statistics for a vector passing results in a structure.
	Example:
		void test()
		{
			RangeResults rRanR;
			vector vD = {1,2,3,4,5,6,7,8,8,9,9,0,0,12.099,12.098,12};
			int nRet  = stats_range(vD, rRanR);
		}	
	Parameters:
		vData=Input vector containing data for which the range is computed
		rr=Output structure containing range results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_range(const vector& vData, RangeResults& rr);


// Category last updated by Joseph on 09/11/2007
/** >Statistics>Advanced
       Create a frequency count of data in a vector. If a non-NULL pointer is passed for
       vCumulativeCount then the cumulative count is returned as well.
   Example:
		void test()
		{
			int nRet;   
			int nOption = FC_NUMINTERVALS;
			UINT nSourceSize = 5;   //Set the DataSize to 5
			vector vSource;
			vSource.SetSize(nSourceSize);	
			vector			vBinCenters;
			vector			vAbsoluteCounts;
			vector			vCumulativeCounts;
			UINT	nBinCenterSize, nAbsoluteCountSize, nCumulativeCountSize;
			vector 	vBinCentersChk; 	
			vector	vAbsoluteCountsChk,vCumulativeCountsChk;	
			int ii;
			vSource.Data(1, nSourceSize,1);	
			FreqCountOptions fcoOptions;	
			fcoOptions.FromMin = 0;
			fcoOptions.ToMax = 7;
			fcoOptions.StepSize = 7;
			fcoOptions.IncludeLTMin = 0;
			fcoOptions.IncludeGEMax = 0;
			UINT nSize;
			nSize = fcoOptions.StepSize;
			nBinCenterSize = nSize;
			vBinCenters.SetSize(nBinCenterSize);
			nAbsoluteCountSize = nSize;
			vAbsoluteCounts.SetSize(nAbsoluteCountSize);
			nCumulativeCountSize = nSize;
			vCumulativeCounts.SetSize(nCumulativeCountSize);	
			nRet = stats_frequency_count(vSource, fcoOptions, vBinCenters,	vAbsoluteCounts, vCumulativeCounts, nOption);
		} 
   Parameters:
       vDataIn=Input data in vector to be binned
       fcoOptions=Input structure containing binning parameters and options
       vBinCenters=Output vector containing bin centers
       vAbsoluteCounts=Output vector containing frequency counts
       vCumulativeCounts=Output vector containing cumulative frequency counts
   Return:
       Returns STATS_NO_ERROR on success and a non-zero STATS error code on failure.
*/

/// YuI 11/02/04 V7.5158 QA70-7122 OC_DATA_PLOT_CREATION
//	int stats_frequency_count(const vector& vDataIn, const FreqCountOptions& fcoOptions, vector& vBinCenters,
//	   vector& vAbsoluteCounts, vector* vCumulativeCounts = NULL, int nOption = FC_STEPSIZE); 
/// Fisher 12/08/08         QA80-12735       SUPPORT_USER_SUPPLIED_BIN_VECTOR_IN_FREQCOUNTS
int stats_frequency_count(const vector& vDataIn, const FreqCountOptions& fcoOptions, vector& vBinCenters,
	   vector& vAbsoluteCounts, vector& vCumulativeCounts = NULL, int nOption = FC_STEPSIZE,
	   vector& vBins = NULL); 
//int stats_frequency_count(const vector& vDataIn, const FreqCountOptions& fcoOptions, vector& vBinCenters,
//	   vector& vAbsoluteCounts, vector& vCumulativeCounts = NULL, int nOption = FC_STEPSIZE); 
/// End SUPPORT_USER_SUPPLIED_BIN_VECTOR_IN_FREQCOUNTS

	   
/// end OC_DATA_PLOT_CREATION
   
	
   
   
/////////////////////////////////////////////////////////////////////////////////
//
//				Correlation and Regression									
//
/////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Analysis>Advanced
		General Linear Regression with vector inputs.Function ocmath_multiple_regression 
		is the main computational engine.
	Parameters:
		vX=Input independent vector
		vY=Input dependent vector
		vWeight=Input weight vector. If no weight, must set vWeight = 1
		sLROptions=Input struct for linear regression options 
		psFitParameter=Output array of two structs to receive intercept[0] and slope[1] and associated errors
		psRegStats=Optional output pointer to struct containing Regression statistics
		psRegANOVA=Optional output pointer to struct containing ANOVA statistics.
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_linear_fit(const vector& vX, const vector& vY, const vector& vWeightIn,
	const LROptions& sLROptions, FitParameter* psFitParameter, // [0] = intercept and [1] = slope
	RegStats* psRegStats = NULL, RegANOVA* psRegANOVA = NULL, RegCalcInternal* psInternal = NULL, // General Linear Regression with vector inputs.
	 matrix& mCov=NULL, matrix& mCorr=NULL);
	 
// Category last updated by Echo on 03/22/2007
/** >Analysis>Advanced
		Function to perform multiple linear regression. Function ocmath_multiple_linear_regression 
		is the main computational engine.The function finds a QR decomposition of X. If R is of 
		full rank, solution is obtained from the QR decomposition. If R is not full rank, 
		a solution is obtain by means of a singular value decomposion(SVD) of R If R is 
		not full rank, STATS_ERROR_RANK_DEFICIENT is returned in addition to the solutions. 
	Example:
		void tt()
		{
			int nRet;
			
			UINT nPts,nM;
			uint nSizeFitParams;
			LROptions psLROptions;
			RegStats psRegStats;
			RegANOVA psRegANOVA;
		   matrix mX_1(5,3)={{1,   2,   3},
   							 {3.2, 1.1, 3.4},
   							 {1,   2.6, 5},
   							 {5,   5,   2.1},
   							 {1,   6,   7}};
		   vector vY_1 ={4,2,6, 7, 8};
		   vector vWeight_1 = {0.2, 0.6, 0.8, 0.3, 1}; 
		   psLROptions.FixIntercept = FALSE;
		   psLROptions.Confidence = 0.95;
		   FitParameter psFitParameter_1[4];
		   nRet = stats_multiple_linear_regression(mX_1, vY_1, vWeight_1, psLROptions, psFitParameter_1, 4, &psRegStats, &psRegANOVA);
		}	
	Parameters:
		mX=Input matrix containing data points of the independent variables
		vY=Input vector containing data points of dependent variable
		vWT=Input vector containing weights of data points, NULL if weighting not used
		sLROptions=Input structuure for linear regression options 
		psFitParameter=Output array of structs to receive information on each fitting parameter
		nSizeFitParams= size (number of structs) of psFitParameter 
		psRegStats=Optional output pointer to struct containing Regression statistics
		psRegANOVA=Optional output pointer to struct containing ANOVA statistics.
		mCov = Optional output coriviance matrix of estimate
		mCorr = Optional Output correlation matrix of estimate
		vH = Optional output diagonal element of leverage, H
		vSE = Optional output standard error of mean predicted values
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/

int stats_multiple_linear_regression(const matrix& mX, const vector& vY, vector& vWT, const LROptions& sLROptions,
		FitParameter* psFitParameter, uint nSizeFitParams, RegStats* psRegStats, RegANOVA* psRegANOVA, 
		matrix& mCov=NULL, matrix& mCorr=NULL, vector& vH = NULL, vector& vSE = NULL);


// Category last updated by Echo on 03/22/2007
/** >Analysis>Advanced
		Function to perform polynomial. Function ocmath_multiple_linear_regression 
		is the main computational engine.
	Example:
		void tt()
		{
			int nOrder,nRet;	
			LROptions sLROptions;	
			RegStats sRegStats;	
			RegANOVA sRegANOVA;
			vector vX_2 = { 0.1, 2.3, 3.1, 4.2 };
			vector vY_2 = { 2.6, 2.8, 4.5, 3.7 };
			vector vWeightIn_2 = { 0.2, 0.8, 1.1, 0.7 };						
			sLROptions.Confidence = 0.95;	
			nOrder = 2;		
			FitParameter sFitParameter_2[3];  			
			matrix mCov(3, 3) ;
			matrix mCorr(3, 3);			
			nRet = stats_polynomial_fit(vX_2,vY_2,vWeightIn_2,nOrder,sLROptions,sFitParameter_2,3,&sRegStats,&sRegANOVA, mCov, mCorr);
		}	
	Parameters:
		vX=Input vector containing data points of the independent variables
		vY=Input vector containing data points of dependent variable
		vWeightIn=Input vector containing weights of data points, NULL if weighting not used
		nOrder=Input order of polynormial funcion. 
		sLROptions=Input structuure for linear regression options 
		psFitParameter=Output array of structs to receive information on each fitting parameter
		nSizeFitParams=size (number of structs) in psFitParameter, must be at least (nOrder+1)
		psRegStats=Optional output pointer to struct containing Regression statistics
		psRegANOVA=Optional output pointer to struct containing ANOVA statistics.
		mCov = Optional output coriviance matrix of estimate
		mCorr = Optional Output correlation matrix of estimate
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_polynomial_fit(const vector& vX, const vector& vY, const vector& vWeightIn,
		int nOrder, const LROptions& sLROptions, FitParameter* psFitParameter, uint nSizeFitParams, 
		RegStats* psRegStats, RegANOVA* psRegANOVA, matrix& mCov=NULL, matrix& mCorr=NULL, vector& vH = NULL); ///Arvin 12/12/06 ADD_RESIDUAL_ANALYSIS_FOR_FITTING


// Category last updated by Echo on 03/22/2007
/**$
		Function to caculate values for LCI, UCI, LPI, UPI in regression.
	Parameters:
		vX=Input vector containing data points of the independent variables
		vY=Input vector containing data points of dependent variable
		psInternal=Output structure to values for confidence interval and predition interval computation. 
		vWeight=Input pointer to vector containing weights of data points, NULL if weighting not used
*/
int stats_calc_internal(const vector& vX, const vector& vY, RegCalcInternal& sInternal, const vector& vWeight = NULL);


// Category last updated by Echo on 03/22/2007
/**$
   		 Function to find if exists duplicated x for given y.
   	Example:
		void tt()
		{
			vector vX = {1, 2, 3, 4, 5};
			vector vY = {2, 4, 6, 4, 2};
			Dataset dsX = vX;
			Dataset dsY = vY;
			double dY = 3.5;
			bool bCheck;	
			bCheck = check_duplicate_x(dY, dsX, dsY);	
		}
	Parameter:
		dX=Input given X value 
		dsSrcX = Input vector to the X value of the curve, must be in strictly ascending order, otherwise it should be sorted before calling this function. size is nSrcSize.
		dsSrcY = Input vector to the Y value of the curve. size is nSrcSize.
  	Return:
    	Returns true if duplicated x for given y, otherwise return false.
*/
bool check_duplicate_x(const double dY, const Dataset dsSrcX, const Dataset dsSrcY);


// Category last updated by Echo on 03/22/2007
/**$
   		 Function to compute a Y value from a standard curve given X.
   	Example:
		void tt()
		{
			vector vX = {1, 2, 3, 4, 5};
			vector vY = {2, 4, 6, 4, 2};
			Dataset dsX = vX;
			Dataset dsY = vY;
			double dY = 3.5;
			double dXLo = 1;
			double dXUp = 3;
			double dX = compute_x_from_y(dY, dXLo, dXUp, dsX, dsY);	
		}   		
  	Parameters:
		ifault = Output value indicates error returned. 
  		dY=Input given Y value 
		dXLo=Input Lower bounding X value of X interval containing Y value
    	dXUp=Input Upper bounding X value of X interval containing Y value
		dsSrcX = Input vector to the X value of the curve, must be in strictly ascending order, otherwise it should be sorted before calling this function. size is nSrcSize.
		dsSrcY = Input vector to the Y value of the curve. size is nSrcSize.
		nMode =Input  interpolation method. Must be one of the three modes: 
				INTERP_TYPE_LINEAR(linear interpolation), 
				INTERP_TYPE_SPLINE(cubic spline interpolation with natural boundary condition), 
				INTERP_TYPE_BSPLINE(B-Spline curve fitting using method by Dierckx.P)
		dSmoothingFactor = (Input)This argument specifies the closeness to the original data. It is only useful when nMode = INTERP_TYPE_BSPLINE. dSmoothingFactor >= 0.
				By means of this parameter, the user can control the tradeoff between closeness of fit and smoothness of fit of the approximation.
				If dSmoothingFactor is too large, the spline will be too smooth and signal will be lost ; if it is too small the spline will pick up too much noise. 
				In the extreme cases the program will return an interpolating spline if  dSmoothingFactor=0 and the weighted least-squares polynomial of degree 3 if s is very large.			
		pWeights = pointer to weights, which is only used in method INTERP_TYPE_BSPLINE, by default(pWeights = NULL) all weights are 1. size is nSrcSize
  	Return:
    	Returns the Y value for the given X value.
*/
double compute_x_from_y(int& ifault, const double dY, const double dXLo, const double dXUp, const Dataset dsSrcX, const Dataset dsSrcY, int nMode = INTERP_TYPE_LINEAR, double dSmoothingFactor = 0.0, vector* pvWeights = NULL);



// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Correlation Coefficient function used to compute the Pearson Product Moment 
		correlation coefficient. 
	Example:
		void test_pearson()
		{
			vector vData = {0.86528,0.39649,0.44901,0.77041,0.43139,0.11343,0.12061,0.00387,0.21913,0.66058,0.31848,0.00595,0.80696,0.48388,0.74349,0.02226,0.59422,0.37465,0.38224,0.12831,0.01272,0.86464,0.42911,0.48951,0.96697,0.62581,0.65671,0.03253,0.7749,0.99037,
					0.1638,0.44968,0.87167,0.66712,0.8677,0.24795,0.26741,0.33618,0.51313,0.34292,0.35445,0.15694,0.51422,0.87666,0.3459,0.71896,0.43328,0.76378,0.42619,0.88064,0.9931,0.55338,0.21309,0.74246,0.33791,0.08578,0.52487,0.43694,0.6661,0.11727,
					0.40405,0.03703,0.82496,0.3849,0.29909,0.2865,0.23733,0.27017,0.59356,0.79517,0.46044,0.62249,0.63896,0.85607,0.48609,0.75937,0.90884,0.44613,0.98352,0.26523};
		
			vector<int> vLevels={30,30,20};
				
			matrix mCorr(vLevels.GetSize(), vLevels.GetSize());
			matrix mSig(vLevels.GetSize(), vLevels.GetSize());
			
			int nRet = stats_pearson_corr_coeff(vData, vLevels, mCorr, mSig);
			
			MatrixPage mp;
			mp.Create("Origin");
			mp.Rename("Corr");
			Matrix mat(mp.GetName());
			mat = mCorr;
			mp.Create("Origin");
			mp.Rename("Significance");
			Matrix mat1(mp.GetName());
			mat1 = mSig;
			
			//The Output is as following:
			//mCorr = {1	-0.19034	0.2054
					//{-0.19034	1	0.14325}
					//{0.2054	0.14325	1}}
					//
			//mSig = {{NANUM	0.3137	0.38497}
					//{0.3137	NANUM	0.54685}
					//{0.38497	0.54685	NANUM}}	
		}	
	Parameters:
		vData=Input vector containing all observations in each group. Size of vData should not be less than 4.
		vIndex=Input vector containing the data number of each group. Size of vIndex should not be less than 2. Each value of vIndex should not be less than 2 
		mCorr=Output matrix containing correlation coefficients. Diagonal of mCorr should be 1.
		mSig=Output matrix containing probability matrix. Diagonal of mCorr should be NANUM.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
///Joseph 06/26/07 NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE	 
//int stats_pearson_corr_coeff(const vector& vData, const vector<int>& vIndex, matrix& mCorr, matrix& mSig);
///End NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Correlation Coefficient function used to compute the Spearman rank correlation coefficient.
	Example:
		void test_spear()
		{
			vector vData = {0.86528,0.39649,0.44901,0.77041,0.43139,0.11343,0.12061,0.00387,0.21913,0.66058,0.31848,0.00595,0.80696,0.48388,0.74349,0.02226,0.59422,0.37465,0.38224,0.12831,0.01272,0.86464,0.42911,0.48951,0.96697,0.62581,0.65671,0.03253,0.7749,0.99037,
					0.1638,0.44968,0.87167,0.66712,0.8677,0.24795,0.26741,0.33618,0.51313,0.34292,0.35445,0.15694,0.51422,0.87666,0.3459,0.71896,0.43328,0.76378,0.42619,0.88064,0.9931,0.55338,0.21309,0.74246,0.33791,0.08578,0.52487,0.43694,0.6661,0.11727,
					0.40405,0.03703,0.82496,0.3849,0.29909,0.2865,0.23733,0.27017,0.59356,0.79517,0.46044,0.62249,0.63896,0.85607,0.48609,0.75937,0.90884,0.44613,0.98352,0.26523};
		
			vector<int> vLevels={30,30,20};
				
			matrix mCorr(vLevels.GetSize(), vLevels.GetSize());
			matrix mSig(vLevels.GetSize(), vLevels.GetSize());
			
			int nRet = stats_spearman_corr_coeff(vData, vLevels, mCorr, mSig);
			
			MatrixPage mp;
			mp.Create("Origin");
			mp.Rename("Corr");
			Matrix mat(mp.GetName());
			mat = mCorr;
			mp.Create("Origin");
			mp.Rename("Significance");
			Matrix mat1(mp.GetName());
			mat1 = mSig;
			
			//The Output is as following:
			//mCorr = {{1	-0.11368	0.29925}
					//{-0.11368	1	0.15639}
					//{0.29925	0.15639	1}}
					//
			//mSig = {{NANUM	0.54974	0.19994}
					//{0.54974	NANUM	0.51026}
					//{0.19994	0.51026	NANUM}	
		}

	Parameters:
		vData=Input vector containing all observations in each group. Size of vData should not be less than 4.
		vIndex=Input vector containing the data number of each group. Size of vIndex should not be less than 2. Each value of vIndex should not be less than 2 
		mCorr=Output matrix containing correlation coefficients. Diagonal of mCorr should be 1.
		mSig=Output matrix containing probability matrix. Diagonal of mCorr should be NANUM.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
///Joseph 06/26/07 NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE	 
//int stats_spearman_corr_coeff(const vector& vData, const vector<int>& vIndex, matrix& mCorr, matrix& mSig);
///End NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Correlation Coefficient function used to compute the Kendall rank correlation coefficient.
	Example:
		void stats_kendall_corr_coeff_Example1()
		{
			vector vData = {0.86528,0.39649,0.44901,0.77041,0.43139,0.11343,0.12061,0.00387,0.21913,0.66058,0.31848,0.00595,0.80696,0.48388,0.74349,0.02226,0.59422,0.37465,0.38224,0.12831,0.01272,0.86464,0.42911,0.48951,0.96697,0.62581,0.65671,0.03253,0.7749,0.99037,
					0.1638,0.44968,0.87167,0.66712,0.8677,0.24795,0.26741,0.33618,0.51313,0.34292,0.35445,0.15694,0.51422,0.87666,0.3459,0.71896,0.43328,0.76378,0.42619,0.88064,0.9931,0.55338,0.21309,0.74246,0.33791,0.08578,0.52487,0.43694,0.6661,0.11727,
					0.40405,0.03703,0.82496,0.3849,0.29909,0.2865,0.23733,0.27017,0.59356,0.79517,0.46044,0.62249,0.63896,0.85607,0.48609,0.75937,0.90884,0.44613,0.98352,0.26523};
			vector<int> vLevels={30,30, 20};
				
			matrix mCorr(vLevels.GetSize(), vLevels.GetSize());
			matrix mSig(vLevels.GetSize(), vLevels.GetSize());
			
			int nRet = ocmath_ken_corr_coeff(vData, vData.GetSize(), vLevels.GetSize(), vLevels, mCorr, mSig);
			
			MatrixPage mp;
			mp.Create("Origin");
			mp.Rename("Corr");
			Matrix mat(mp.GetName());
			mat = mCorr;
			mp.Create("Origin");
			mp.Rename("Significance");
			Matrix mat1(mp.GetName());
			mat1 = mSig;
			
			//The Output is as following:
			//mCorr = {(1	-0.09425	0.16842)
					//{-0.09425	1	0.09474}
					//{0.16842	0.09474	1}}
					//
			//mSig = {{NANUM	0.46448	0.29917}
					//{0.46448	NANUM	0.55922}
					//{0.29917	0.55922	NANUM}}
		}

	Parameters:
		vData=Input vector containing all observations in each group. Size of vData should not be less than 4.
		vIndex=Input vector containing the data number of each group. Size of vIndex should not be less than 2. Each value of vIndex should not be less than 2 
		mCorr=Output matrix containing correlation coefficients. Diagonal of mCorr should be 1.
		mSig=Output matrix containing probability matrix. Diagonal of mCorr should be NANUM.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
///Joseph 06/26/07 NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE
//don't use these three functions in oc by Echo said. 
/*
int stats_kendall_corr_coeff(const vector& vData, const vector<int>& vIndex, matrix& mCorr, matrix& mSig);
*/		
///End NOT_USE_THIS_FUNCTION_IN_OC_ANY_MORE
/////////////////////////////////////////////////////////////////////////////////
//
//				t-Test Functions									
//
/////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a One Sample t-Test.
	Example:
		void tt()
		{
			vector<double> vTest;	
			HypotTestOptions Hypo;
			tTestResults tRes;
			tTestDescStats DescRes;
			int nSize = 30;
			vTest.SetSize(nSize);	
			for(int i=0;i<nSize;i++)
				vTest[i]=i*i/10.0;	
			int nRet;
			Hypo.TailType=TAILED_TEST_TWO;
			Hypo.HypotValue=4.0;		
			nRet=stats_one_sample_t_test(vTest, Hypo, DescRes, tRes);
		}		
	Parameters:
		vData=Input vector containing data points for One Sample t-Test
		opt=Input structure containing t-Test options
		DescRes=Output structure containing Descriptive statistics of tTest.
		tTestRes=Output structure containing test result of tTest
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_one_sample_t_test(const vector& vData, const HypotTestOptions& opt,
	 tTestDescStats& DescRes, tTestResults& tTestRes);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute confidece levels for a mean.
	Example:
		void tt()
		{
			int nRet;	
			vector <double>vLevels;	
			tTestMean1SampleResults tRes;
			vector vLCIs;
 			vector vUCIs; 	
 			tRes.N=10;
 			tRes.Mean=2.85	;
 			tRes.SD=2.83049	;
 			tRes.SEM=0.89508 ;
 			tRes.DOF=9;
 			tRes.tValue=-1.2848 ;
 			tRes.Prob=0.23094;	
 			vLevels.SetSize(3);
 			vLevels[0]=0.7;
 			vLevels[1]=0.8;
 			vLevels[2]=0.9;
 			vLCIs.SetSize(vLevels.GetSize());
 			vUCIs.SetSize(vLevels.GetSize()); 	
 			nRet=stats_confidence_levels_for_mean(vLevels,tRes,vLCIs,vUCIs);
		}		
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}
		tTestRes=Input structure containing t-Test results from stats_one_sample_t_test
		vLCIs=Output vector containing Lower Confidence Limits for mean
		vUCIs=Output vector containing Upper Confidence Limits for mean
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_confidence_levels_for_mean(const vector vLevels, const tTestMean1SampleResults& tTestRes,
	 vector& vLCIs, vector& vUCIs);

// Category last updated by Joseph on 09/11/2007
/**$ >Statistics>Advanced
		Function to perform a Chi-Square Test for equal variance.
	Example:
		void test()
		{
			HypotTestOptions HTO;
			tTestMean1SampleResults tRes;
			ChiSqVar1SampleResults csvRes;	
			tRes.N=10;
 			tRes.Mean=2.85	;
 			tRes.SD=2.83049	;
 			tRes.SEM=0.89508 ;
 			tRes.DOF=9;
 			tRes.tValue=-1.2848 ;
 			tRes.Prob=0.23094;
 			int nRet; 	
 			ChiSqVar1SampleResults Chk_csvRes; 	
			HTO.TailType=TAILED_TEST_TWO;
			HTO.HypotValue=2;	
			Chk_csvRes.ChiSqDOF=9;
			Chk_csvRes.ChiSqValue=36.05253;
			Chk_csvRes.ChiSqProb=0.0000776102;	
			nRet=stats_chi_square_test(HTO,tRes,csvRes);	
		}		
	Parameters:
		opt=Input structure containing Chi-Square Test options
		tTestRes=Input structure containing t-Test results output from stats_one_sample_t_test
		res=Output structure containing Chi-Square Test Results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
/*
int stats_chi_square_test(const HypotTestOptions& opt, const tTestMean1SampleResults& tTestRes,
	 ChiSqVar1SampleResults& res);
*/

// Category last updated by Joseph on 09/11/2007
/**$ >Statistics>Advanced
		Function to compute confidece levels for a variance.
	Example:
		void test()
		{
			int nRet;
			vector <double>vLevels;	
			tTestMean1SampleResults tRes;
			vector vLCIs;
 			vector vUCIs;	
 			tRes.N=10;
 			tRes.Mean=2.85	;
 			tRes.SD=2.83049	;
 			tRes.SEM=0.89508 ;
 			tRes.DOF=9;
 			tRes.tValue=-1.2848 ;
 			tRes.Prob=0.23094;
 			vLevels.SetSize(3);
 			vLevels[0]=0.7;
 			vLevels[1]=0.8;
 			vLevels[2]=0.9;
 			vLCIs.SetSize(vLevels.GetSize());
 			vUCIs.SetSize(vLevels.GetSize());	
 			nRet=stats_confidence_levels_for_variance(vLevels,tRes,vLCIs,vUCIs);
		}		
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}
		tTestRes=Input structure containing t-Test results from stats_one_sample_t_test
		vLCIs=Output vector containing Lower Confidence Limits for variance
		vUCIs=Output vector containing Upper Confidence Limits for variance
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/

/*
int stats_confidence_levels_for_variance(const vector vLevels, const tTestMean1SampleResults& tTestRes,
	 vector& vLCIs, vector& vUCIs);
*/

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute actual or hypothetical powers for a one sample t-Test.
	Example:
		void test()
		{
			double dSigLevel;
			int iTailType;
			vector vSampleSizes;
			tTestMean1SampleResults tRes;
			tTestDescStats DescRes;
			tTestResults tTres;
			vector<double> vPowers;	
			tRes.N=10;
 			tRes.Mean=2.85	;
 			tRes.SD=2.83049	;
 			tRes.SEM=0.89508 ;
 			tRes.DOF=9;
 			tRes.tValue=-1.2848 ;
 			tRes.Prob=0.23094;	
 			int nRet;
			dSigLevel=0.05;
			iTailType=TAILED_TEST_TWO;	
			vSampleSizes.SetSize(3);
			vSampleSizes[0]=2;
			vSampleSizes[1]=5;
			vSampleSizes[2]=6;	
			vPowers.SetSize(vSampleSizes.GetSize());	
			nRet=stats_tTest_powers(dSigLevel,iTailType,vSampleSizes,tRes,vPowers);
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		iTailType=Input number indicates compute powers for a two-tailed, upper tailed, or lower tailed test
		vSampleSizes=Input vector containing actual and hypothetical sample sizes
		tTestRes=Input structure containing t-Test results output from stats_one_sample_t_test
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_tTest_powers(const double dSigLevel, const int iTailType, const vector& vSampleSizes, // Use vector double (vSampleSizes) for convenience
	 tTestMean1SampleResults& tTestRes, vector& vPowers);                                             //   as string::GetTokens requires doubles

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to preprocess Descriptive Statistics results in two sample ttest as required
	Parameter:
		DescStats1=Input structure containing Descriptive statistics of group1
		DescStats2=Input structure containing Descriptive statistics of group2
		DescStatsDiff=Iput structure containing Descriptive statistics of difference of group1 and group2
		tTestRes=Input structure containing t-Test Results
		res=Output structure containing t-Test results from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
*/
void stats_set_ttest2_results(const tTestDescStats& DescStats1, const tTestDescStats& DescStats2, 
								const tTestDescStats& DescStatsDiff, const tTestResults& tTestRes, tTestMean2SampleResults& res);
								
// Category last updated by Echo on 03/22/2007
/**$
		Function to preprocess Descriptive Statistics results in two sample ttest as required
	Parameter:
		res=Input structure containing t-Test results from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		DescStats1=Output structure containing Descriptive statistics of group1
		DescStats2=Output structure containing Descriptive statistics of group2
		DescStatsDiff=Output structure containing Descriptive statistics of difference of group1 and group2
		tTestRes=Output structure containing t-Test Results
*/
void stats_get_ttest2_results(const tTestMean2SampleResults& res, tTestDescStats& DescStats1, tTestDescStats& DescStats2, 
								tTestDescStats& DescStatsDiff, tTestResults& tTestRes);
	
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Two Sample t-Test for equal variance.
	Example:
		
		void test()
		{
			int nRet;
			vector vData1,vData2;
			HypotTestOptions opt;
			tTestDescStats DescStats1,DescStats2,DescStatsDiff;
			tTestResults tTestRes;
			int nDataSize = 5;	
			vData1.SetSize(nDataSize);
			vData2.SetSize(nDataSize);
			vData1[0] =  1.7;	vData2[0] = 3.2;
			vData1[1] =  2.1;	vData2[1] = 3.9;
			vData1[2] =  3.9;	vData2[2] = 4.9;
			vData1[3] =  7.2;	vData2[3] = 5.3;
			vData1[4] =  8.6;	vData2[4] = 5.5;
			opt.HypotValue = 0;
			opt.TailType = TAILED_TEST_TWO;
			nRet = stats_two_sample_t_test(vData1,vData2,opt,DescStats1,DescStats2,DescStatsDiff,tTestRes);
		}
		
	Parameters:
		vData1=Input vector containing 1st data points for Two Sample t-Test
		vData2=Input vector containing 2nd data points for Two Sample t-Test
		opt=Input structure containing t-Test options
		DescStats1=Output structure containing Descriptive statistics of group1
		DescStats2=Output structure containing Descriptive statistics of group2
		DescStatsDiff=Output structure containing Descriptive statistics of difference of group1 and group2
		tTestRes=Output structure containing t-Test Results
		tTestResUneqVar=Output structure contraining t-Test results(assumed unequal variance) 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_t_test(const vector& vData1, const vector& vData2, const HypotTestOptions& opt,
	 tTestDescStats& DescStats1, tTestDescStats& DescStats2, tTestDescStats& DescStatsDiff, tTestResults& tTestRes, tTestResults& tTestResUneqVar = NULL);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Two Sample t-Test for unequal variance.
	Parameters:
		vData1=Input vector containing 1st data points for Two Sample t-Test
		vData2=Input vector containing 2nd data points for Two Sample t-Test
		opt=Input structure containing t-Test options
		DescStats1=Output structure containing Descriptive statistics of group1
		DescStats2=Output structure containing Descriptive statistics of group2
		DescStatsDiff=Output structure containing Descriptive statistics of difference of group1 and group2
		tTestRes=Output structure containing t-Test Results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/	 
int stats_two_sample_t_test_unequal_var(const vector& vData1, const vector& vData2, const HypotTestOptions& opt,
	 tTestDescStats& DescStats1, tTestDescStats& DescStats2, tTestDescStats& DescStatsDiff, tTestResults& tTestRes);	 

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute confidence levels for the mean of vData1 minus the mean of vData2 for equal variance.
	Example:
		void tt()
		{
		   vector vData1 = { 1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
		   vector vData2 = { 3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };
		   vector vLevels = { 90, 95, 99 };
		   HypotTestOptions opt;
		   tTestMean2SampleResults res;
		   tTestResults tTestRes;
		   tTestDescStats DescStats1,DescStats2,DescStatsDiff;
		   vector vLCIs, vUCIs;
		   double lower_limit, upper_limit;
		   double nSize1 = vData1.GetSize();
		   double nSize2 = vData2.GetSize();
		   opt.TestType = 0;
		   opt.HypotValue = 0;
		   opt.TailType = 0;
		   stats_two_sample_t_test(vData1, vData2, opt, DescStats1 ,DescStats2, DescStatsDiff, tTestRes);
		   stats_two_sample_confidence_levels_for_mean_equal_var(vLevels, res, vLCIs, vUCIs);
		   ASSERT ( round ( vLCIs[0], 5 ) == -3.29459 );
		   ASSERT ( round ( vLCIs[1], 5 ) == -3.66286 );
		   ASSERT ( round ( vLCIs[2], 5 ) == -4.43699);
		   ASSERT ( round ( vUCIs[0], 5 ) == 0.22187);
		   ASSERT ( round ( vUCIs[1], 5 ) == 0.59013);
		   ASSERT ( round ( vUCIs[2], 5 ) == 1.36426);
		} 
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}		
		tTestRes=Input structure containing t-Test results from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		vLCIs=Output vector containing Lower Confidence Limits for mean
		vUCIs=Output vector containing Upper Confidence Limits for mean
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_confidence_levels_for_mean_equal_var(const vector vLevels, const tTestMean2SampleResults& tTestRes,
	 vector& vLCLs, vector& vUCLs);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute confidece levels for the mean of vData1 minus the mean of vData2 for unequal variance.
	Example:
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}		
		vData1=Input vector containing 1st data points for Two Sample t-Test
		vData2=Input vector containing 2nd data points for Two Sample t-Test
		vLCIs=Output vector containing Lower Confidence Limits for mean
		vUCIs=Output vector containing Upper Confidence Limits for mean
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure
	
*/
int stats_two_sample_confidence_levels_for_mean_unequal_var(const vector vLevels, const tTestMean2SampleResults& tTestRes,
	 vector& vLCLs, vector& vUCLs);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Chi-Square Test for equal variance.
	Example:
		void tt()
		{
			vector vData1 = { 96.3, 96.7, 96.9, 97, 97.1, 97.1, 97.1, 97.2, 97.3 };
			vector vData2 = { 96.4, 96.7, 96.8, 97.2, 97.2, 97.4, 97.6, 97.7, 97.7 };
	
			HypotTestOptions opt;
			tTestMean2SampleResults TestRes;    
			ChiSqVar2SampleResults chi_res;    
			double lower_limit, upper_limit;
			double nSize1 = vData1.GetSize();
			double nSize2 = vData2.GetSize();
			opt.TestType = 0;
			opt.HypotValue = 1;
			opt.TailType = TAILED_TEST_LOWER;    
			stats_two_sample_t_test_equal_var(vData1, vData2, opt, TestRes);
			stats_two_sample_chi_square_test(opt, TestRes, chi_res);

			ASSERT( chi_res.ChiSqDOF == 16 );
			ASSERT( round( chi_res.ChiSqValue, 6 ) == 2.488889 );
			ASSERT( round( chi_res.ChiSqProb, 8 ) == 0.99995242 );
			ASSERT( chi_res.ChiSqDOF == 16 );
			ASSERT( round( chi_res.ChiSqValue, 6 ) == 2.488889 );
			ASSERT( round( chi_res.ChiSqProb, 8 ) == 0.99995242 );
		} 
	Parameters:
		opt=Input structure containing Chi-Square Test options
		tTestRes=Input structure containing t-Test results output from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		res=Output structure containing Chi-Square Test Results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_chi_square_test(const HypotTestOptions& opt, const tTestMean2SampleResults& tTestRes,	 ChiSqVar2SampleResults& res);

// Category last updated by Joseph on 09/11/2007
/**$ >Statistics>Advanced
		Function to perform a F Test for equal variance.
	Example:
		void tt()
		{
			vector vData1 = { 1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
			vector vData2 = { 3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };
			vector vLevels = { 90.0, 95.0, 99.0 };
			HypotTestOptions opt;
			tTestMean2SampleResults res;
			vector vLCIs, vUCIs;
			FVarResults FResult;
			double lower_limit, upper_limit;
			double nSize1 = vData1.GetSize();
			double nSize2 = vData2.GetSize();
			opt.TestType = 0;
			opt.HypotValue = 0;
			opt.TailType = TAILED_TEST_LOWER;
			stats_two_sample_t_test_equal_var(vData1, vData2, opt, res);
			stats_f_test(opt,res, FResult);
			printf("NUM = %d\nDEN = %d\nFValue = %f\nFProb = %f\n", FResult.FDOFNum, FResult.FDOFDen, FResult.FValue, FResult.FProb );
			ASSERT( FResult.FDOFNum == 10 );
			ASSERT( FResult.FDOFDen == 10 );
			ASSERT( round( FResult.FValue, 2 ) == 2.35 );
			ASSERT( round( FResult.FProb, 4 ) == 0.9026 );
		}
	Parameters:
		opt=Input structure containing F Test options
		tTestRes=Input structure containing t-Test results output from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		res=Output structure containing F Test Results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/

/*
int stats_f_test(const HypotTestOptions& opt, const tTestMean2SampleResults& tTestRes, FVarResults& res);
*/

// Comment last updated by Joseph on 09/11/2007
/**$ >Statistics>Advanced
		Function to compute confidece levels for common or pooled variance, based on F test.
	Example:
		void tt()
		{
			vector vData1 = {1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
			vector vData2 = {3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };
			vector vLevels = { 90, 95, 99 };
			vector vLCIs, vUCIs;
			HypotTestOptions opt;
			tTestMean2SampleResults TestRes;    
			ChiSqVar2SampleResults chi_res;    
			double lower_limit, upper_limit;
			double nSize1 = vData1.GetSize();
			double nSize2 = vData2.GetSize();
			opt.TestType = 0;
			opt.HypotValue = 1;
			opt.TailType = TAILED_TEST_LOWER;    
			ocmath_two_sample_t_test_equal_var(vData1, vData2, opt, TestRes);
			stats_two_sample_confidence_levels_for_variance(vLevels, TestRes, vLCIs, vUCIs);
			ASSERT( round( vLCIs[0], 4 ) == 0.7876 );
			ASSERT( round( vUCIs[0], 4 ) == 6.9863 );
			ASSERT( round( vLCIs[1], 4 ) == 0.6311 );
			ASSERT( round( vUCIs[1], 4 ) == 8.7188 );
			ASSERT( round( vLCIs[2], 4 ) == 0.4012 );
			ASSERT( round( vUCIs[2], 3 ) == 13.715 );
		} 
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}		
		tTestRes=Input structure containing t-Test results from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		vLCIs=Output vector containing Lower Confidence Limits for variance
		vUCIs=Output vector containing Upper Confidence Limits for variance
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/	 
/*
int stats_two_sample_confidence_levels_for_variance(const vector vLevels, const tTestMean2SampleResults& tTestRes,
	 vector& vLCIs, vector& vUCIs);
*/
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute actual or hypothetical powers for a two sample t-Test.
	Example:
		void test()
		{
			int nRet;
			double dSigLevel; 
			vector vPowers,vData1,vData2;
			vector vSampleSizes;
			HypotTestOptions opt;			
			tTestDescStats DescStats1, DescStats2, DescStatsDiff;
			tTestMean2SampleResults tMean2SampleRes;		
			tTestResults tTestRes;
			int nSampleSize;	//Size of vSampleSizes
			vector vPowersChk; //vector for checking vPowers
			int ii;
			int nDataSize = 5;
			vData1.SetSize(nDataSize);
			vData2.SetSize(nDataSize);
			vData1[0] =  1.7;	vData2[0] = 3.2;
			vData1[1] =  2.1;	vData2[1] = 3.9;
			vData1[2] =  3.9;	vData2[2] = 4.9;
			vData1[3] =  7.2;	vData2[3] = 5.3;
			vData1[4] =  8.6;	vData2[4] = 5.5;
			opt.HypotValue = 0;
			opt.TailType = TAILED_TEST_TWO;
			nRet = stats_two_sample_t_test(vData1,vData2,opt,DescStats1,DescStats2,DescStatsDiff,tTestRes);
			
			if ( STATS_NO_ERROR == nRet)	//Run successfully(stats_two_sample_t_test_equal_var)
			{	
				dSigLevel = 0.05;
				nSampleSize = 3;
				vSampleSizes.SetSize(nSampleSize);
				vSampleSizes[0] = 50;
				vSampleSizes[1] = 500;
				vSampleSizes[2] = 2000;
				tMean2SampleRes.N1 = DescStats1.N;
				tMean2SampleRes.Mean1 = DescStats1.Mean;
				tMean2SampleRes.SD1 = DescStats1.SD;
				tMean2SampleRes.SEM1 = DescStats1.SEM;
				tMean2SampleRes.N2 = DescStats2.N;
				tMean2SampleRes.Mean2 = DescStats2.Mean;
				tMean2SampleRes.SD2 = DescStats2.SD;
				tMean2SampleRes.SEM2 = DescStats2.SEM;
				tMean2SampleRes.DiffMean = DescStatsDiff.Mean;
				tMean2SampleRes.DiffSEM = DescStatsDiff.SEM;
				tMean2SampleRes.DOF = tTestRes.DOF;
				tMean2SampleRes.tValue = tTestRes.tValue;
				tMean2SampleRes.Prob = tTestRes.Prob;
				nRet = stats_two_sample_tTest_powers(dSigLevel,opt,vSampleSizes,tMean2SampleRes,vPowers);
			}
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vSampleSizes=Input actual and hypothetical sample sizes		
		tTestRes=Input structure containing t-Test results output from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var		
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_powers(const double dSigLevel, const HypotTestOptions& opt, const vector& vSampleSizes,
	 tTestMean2SampleResults& tTestRes, vector& vPowers);
	

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Two Sample t-Test for equal variance.
	Example:
		bool test_stats_two_sample_t_test_pair()
		{
		   int nRet;    
		   bool bRet;
		   //Notice: Size of vData1 and vData2 must be the same 
		   vector vData1 = { 1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
		   vector vData2 = { 3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };     
		   HypotTestOptions opt;                //Type defines in stats_types.h
		   opt.HypotValue = 0;
		   opt.TailType = TAILED_TEST_TWO;
		   tTestMean2SampleResults res;        //Type defines in stats_types.h
		   nRet = stats_two_sample_t_test_pair(vData1,vData2,opt,res);
		   if ( STATS_NO_ERROR == nRet)    //Run successfully
		   {
				ASSERT(0 == 11 - res.N1);
				ASSERT(is_equal(round(res.Mean1,4),round(4.60909,4)));
				ASSERT(is_equal(round(res.SD1,4),round(2.83106,4)));
				ASSERT(0 == 11 - res.N2);
				ASSERT(is_equal(round(res.Mean2,4),round(6.14545,4)));
				ASSERT(is_equal(round(res.SEM1,4),round(0.8536,4)));
				ASSERT(is_equal(round(res.SD2,4),round(1.84844,4)));
				ASSERT(is_equal(round(res.SEM2,4),round(0.55733,4)));
				ASSERT(is_equal(round(res.DiffMean,4),round(-1.53636,4)));
				ASSERT(is_equal(round(res.DiffSEM,4),round(0.29627,4)));
				ASSERT(0 == 10 - res.DOF);
				ASSERT(is_equal(round(res.tValue,4),round(-1.43241,4)));
				ASSERT(is_equal(round(res.Prob,4),round(0.18254,4)));
				bRet=TRUE;
		   }
		   else // Run unsuccessfully
		   {
				ASSERT(0);
				bRet=FALSE;
		   }
   
		   return bRet;
		} 
	Parameters:
		vData1=Input vector containing 1st data points for Two Sample t-Test
		vData2=Input vector containing 2nd data points for Two Sample t-Test
		opt=Input structure containing t-Test options
		res=Output structure containing t-Test Results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_t_test_pair(const vector& vData1, const vector& vData2, const HypotTestOptions& opt,
	 tTestMean2SampleResults& res);	 

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute confidece levels for the mean of vData1 minus the mean of vData2 for equal variance.
	Example:
		bool test_stats_two_sample_confidence_levels_for_mean_pair()    
		{
			int nRet;    
			bool bRet;
			//Notice: Size of vData1 and vData2 must be the same 
			vector vData1 = { 1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
			vector vData2 = { 3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };   
			HypotTestOptions opt;                //Type defines in stats_types.h
			opt.HypotValue = 0;
			opt.TailType = TAILED_TEST_TWO;
			tTestMean2SampleResults tTestRes;        //Type defines in stats_types.h
			nRet = ocmath_two_sample_t_test_pair(vData1,vData2,opt,tTestRes);
			bRet = FALSE;
			if ( STATS_NO_ERROR == nRet)    //Run successfully(stats_two_sample_t_test_pair)
			{    
				vector vLevels={ 0.90 , 0.95 , 0.98 };    //Confidence Intervals
				vector vLCIs,vUCIs;                   
				nRet = ocmath_two_sample_confidence_levels_for_mean_pair(vLevels, tTestRes,vLCIs,vUCIs);
				
				if ( STATS_NO_ERROR == nRet )    //Run successfully(stats_two_sample_confidence_levels_for_mean_pair)
				{
					vector vLCIsChk = { -3.48036, -3.9262, -4.5007 }; //vector for checking vLCIs
					vector vUCIsChk = {  0.40763, 0.85347, 1.42797 }; //vector for checking vUCIs
					
					for (int ii = 0; ii < vLevels.GetSize() ; ii++)
					{
						ASSERT( is_equal( round(vLCIsChk[ii],4), round(vLCIs[ii],4) ) );
						ASSERT( is_equal( round(vUCIsChk[ii],4), round(vUCIs[ii],4) ) );
					}
					bRet = TRUE;
				}
				else        //Run unsuccessfully (stats_two_sample_confidence_levels_for_mean_pair)
					ASSERT(0);
			}
			else    //Run unsuccessfully (stats_two_sample_t_test_pair)
			ASSERT(0);
			return bRet;
		}
	Parameters:
		vLevels=Input vector containing confidence levels, e.g. {0.9, 0.95, 0.99}		
		tTestRes=Input structure containing t-Test results from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var
		vLCIs=Output vector containing Lower Confidence Limits for mean
		vUCIs=Output vector containing Upper Confidence Limits for mean
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_confidence_levels_for_mean_pair(const vector vLevels, const tTestMean2SampleResults& tTestRes,
	 const HypotTestOptions& opt,vector& vLCIs, vector& vUCIs);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute actual or hypothetical powers for a two sample t-Test.
	Example:
		bool test_stats_two_sample_tTest_powers_pair()
		{
			int nRet;    
			bool bRet;
			//Notice: Size of vData1 and vData2 must be the same 
			vector vData1 = { 1.7, 2.1, 3.9, 7.2, 8.6, 8.5, 7.3, 5.1, 2.8, 1.8, 1.7 };
			vector vData2 = { 3.2, 3.9, 4.9, 5.3, 5.5, 6.2, 6.5, 6.9, 7.5, 8.3, 9.4 };
			HypotTestOptions opt;                //Type defines in stats_types.h
			opt.HypotValue = 0;
			opt.TailType = TAILED_TEST_TWO;
			tTestMean2SampleResults tTestRes;        //Type defines in stats_types.h
			nRet = stats_two_sample_t_test_pair(vData1,vData2,opt,tTestRes);
			bRet = FALSE;
			if ( STATS_NO_ERROR == nRet)    //Run successfully(stats_two_sample_t_test_pair)
			{    
				double dSigLevel = 0.05;
				vector vSampleSizes = { 10,20,50 };
				vector vPowers;
				
				nRet = stats_two_sample_tTest_powers_pair(dSigLevel,opt,vSampleSizes,tTestRes,vPowers);
				
				if ( STATS_NO_ERROR == nRet)    //Run successfully(stats_two_sample_tTest_powers_pair)
				{
					vector vPowersChk = { 0.19944 ,0.43704, 0.84928 }; //vector for checking vPowers
					for (int ii = 0; ii < vSampleSizes.GetSize() ; ii++)
					ASSERT( is_equal( round(vPowersChk[ii],4), round(vPowers[ii],4) ) );
					bRet = TRUE;
				}
				else
					ASSERT(0);        //Run unsuccessfully(stats_two_sample_tTest_powers_pair)
			}
			else
				ASSERT(0);        //Run unsuccessfully (stats_two_sample_t_test_pair)
			return bRet;
		} 
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vSampleSizes=Input actual and hypothetical sample sizes		
		tTestRes=Input structure containing t-Test results output from stats_two_sample_t_test_equal_var or stats_two_sample_t_test_unequal_var		
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_powers_pair(const double dSigLevel, const HypotTestOptions& opt, const vector& vSampleSizes,
	 tTestMean2SampleResults& tTestRes, vector& vPowers);
	 
// Category last updated by Echo on 03/22/2007
/**$
       Computes Fisher exact probabilities for a two-way contingency table.
   Parameters:
       irows=Input number of rows in the table.
       icloumns=Input number of columns in the table.
       vTable=Input vector of length irows*icolumns containing the observed counts
			  in the contingency table for Fisher's Exact Test.
	   TableProb=Output probability of the observed table.
	   Prob=Output two-sided p-value for independence of rows and columns. (Probability for Fisher Exact Test)
	   iOption=Input integer that indicates which Algorithm to be used.
			  0:	The Fisher exact test is used.
			  1:	Use hybird approximation to Fisher's exact test.
			  2:	Use total enumeration method.
   Return:
       Returns STATS_NO_ERROR on success and a non-zero STATS error code on failure.
*/
int stats_fisher_exact_test(const int irows, const int icolumns, const vector& vData, double TableProb, double Prob, int iOption); 



	 
/////////////////////////////////////////////////////////////////////////////////
//
//				Normality Test Functions									
//
/////////////////////////////////////////////////////////////////////////////////	 
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Shapiro-Wilk Normality Test.
	Example:
		void test()
		{
			vector TestData={ 3.1, 4.2, 2.2, 3, 6 };
			NormTestResults SWRes;
			int nRet = stats_shapiro_wilk_test(TestData,SWRes,1);  
		}		
	Parameters:
		vData=Input vector containing data points for Normality Test
		SWRes=Output structure containing Shapiro_Wilk Normality Test results
		iOption=Input integer that indicates which Algorithm to be used
			   0:	Algorithm R94, SAS and SPSS used
			   1:	Algorithm AS181, NAG used
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_shapiro_wilk_test(const vector& vData, NormTestResults& SWRes, int iOption);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Lilliefors Normality Test
		The function performs lilliefors modification of the Kolmogorov-Smirnov	test
		to determine if the null hypothesis of composite normality is a	reasonable 
		assumption regarding the population distribution of a random sample X. 
		The Lilliefors test is based on simulation, so the significance level 
		is restricted to 0.01 <= ALPHA <= 0.20 (the region tabularized by Lilliefors).

	Example:
		void test()
		{
			vector TestData={ 3.1,4.2,-2.2,3,5 };
			NormTestResults LFRes;
			int nRet = stats_lilliefors_test(TestData,LFRes);
		}		
	Parameters:
		vData=Input vector containing data points for Normality Test, Size of vData must be greater than 4
		LFRes=Output structure containing Lilliefors Normality Test results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_lilliefors_test(const vector& vData, NormTestResults& LFRes);

// Category last updated by Joseph on 09/11/2007
/** >Statistics>Advanced
		Function to perform a Kolmogorov-Smirnov Normality Test using ocmath_ksone as main engine 
		imsls_d_kolmogorov_one as the computational engine.
	Example:
		void test()
		{
			int nRet=0;
			NormTestResults KSRes1;
			vector vData1={-2,-1,0,1,2};
			nRet=stats_kolmogorov_smirnov_test(vData1, KSRes1);
		}		
	Parameters:
		vData=Input vector containing data points for Normality Test
		KSRes=Output structure containing Kolmogorov-Smirnov Normality Test results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
/// Max 10/13/04 REPLACE_KOLMOGOROV_SMIRNOV_TEST_WITH_NAG_FUNTION
//int stats_kolmogorov_smirnov_test(const vector& vData, NormTestResults& KSRes);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a Kolmogorov-Smirnov Normality Test using nag function nag_1_sample_ks_test() 
	Example:
		void test()
		{
			int nRet=0;
			NormTestResults KSRes1;
			vector vData1={-2,-1,0,1,2};
			int paratype = 0;
			vector par;
			par.SetSize(2);
			
			nRet=stats_kolmogorov_smirnov_test(vData1, paratype, par, KSRes1);
		}	
	Parameters:
		vData=Input vector containing data points for Normality Test
		paratype=Input flag to specify whether mean and var are known or estimated from the data
		         PARA_ESTIMATED: par would be ouput with estimated mean and var
		         PARA_SUPPLIED:  par need to be supplied with known mean and var.
		par=Input/output vector include the mean and variance. size of par should be 2. 
		KSRes=Output structure containing Kolmogorov-Smirnov Normality Test results
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_kolmogorov_smirnov_test(const vector& vData, const int paratype, vector& par, NormTestResults& KSRes);
/// End REPLACE_KOLMOGOROV_SMIRNOV_TEST_WITH_NAG_FUNTION



/////////////////////////////////////////////////////////////////////////////////
//
//				One Way ANOVA Functions									
//
/////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to get and pre-process all required input datasets
	Example:
		void test_get_data()
		{
			Matrix mat("Matrix2");
			vector vData;
			int iLevels;
			vector<int> vIndex;
			int nRet = stats_anova_get_data_in_multiple_columns(mat, vData, iLevels, vIndex);
		}
	Parameters:
		mData = Input datasets, each column of matrix is a group of data
		vData = Output dataset in a vector
		iLevels = Output number of groups
		vIndex = Output vector containing the data number of each group
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.	
*/
int stats_anova_get_data_in_multiple_columns(const matrix mData, vector& vData, int& iLevels, vector<int>& vIndex);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to get and pre-process all required input datasets
	Example:
		void test_get_data_in_group()
		{
			vector vOrgData = {23, 12, 26, 24, 18};
			vector vGroup = {3, 1, 3, 2, 1};
			vector vData;
			vector<int> vIndex;
			int iLevels;
			stats_anova_get_data_in_group(vOrgData, vGroup, vData, iLevels, vIndex);
		}
	Parameters:
		vOrgData = Input datasets containing data points.
		vGroup = Input dataset containing classification.
		vData = Output dataset in a vector
		iLevels = Output number of groups
		vIndex = Output vector containing the data number of each group
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.	
*/
int stats_anova_get_data_in_group(const vector& vOrgData, const vector& vGroup, vector& vData, int& iLevels, vector<int>& vIndex);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to perform a One-Way ANOVA. Function ocmath_anova_one_way is the main computational engine.
	Example:
		void test()
		{
			int nRet;
			int iLevels;
			ANOVAStats TotalStats;
			RegANOVA ANOVATable;
			vector vData_1 ={77.7111,88.07729,92.15046,77.70871,72.25362,  //Group 1
							 83.52885,75.17097,87.96739,91.59955,81.93563, //Group 2
							 78.15473,79.42937,83.52293,71.4648,72.90409 }; //Group 3
			vector<int> vIndex_1 = { 5,5,5 };
			iLevels = vIndex_1.GetSize();
			ANOVADescStats DescStats_1[3];
			nRet = stats_anova_one_way(vData_1,iLevels,vIndex_1,DescStats_1,3,TotalStats,&ANOVATable);
		}		
	Parameters:
		vData=Input vector containing data points
		iLevels=Input number of groups
		vIndex=Input vector containing the data number of each group.
		DescStats=Output reference to struct containing descriptive statistics of each group.
		nSizeDescStat=Input size (number of structs) of DescStats, which should be less than iLevel.
		ANOVAstats=Output reference to struct containing ANOVA statistics
		ANOVATable=Output reference to struct containing ANOVA table
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_one_way(const vector& vData,
						const int 	iLevels,
						const vector<int>& vIndex,
						ANOVADescStats* DescStats, uint nSizeDescStat,
						ANOVAStats& TotalStats,
						RegANOVA 	*ANOVATable);
						
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute One-Way ANOVA Mean Comparisons. Function ocmath_anova_mean_comparison
		is the main computational engine. Computes simultaneous confidence interval for the differences 
		between means. 
	Example:
		void test()
		{
			int nRet;
			double dAlpha = 0.05;
			int nIntervalType;
			RegANOVA ANOVATable;
			ANOVAStats TotalStats;
			int iLevels;	
			iLevels = 4;
			ANOVATable.AnovaTable[0].DOF = 3;   ANOVATable.AnovaTable[0].SSq = 308.7391453;	ANOVATable.AnovaTable[0].MeanSq = 102.913048;
			ANOVATable.AnovaTable[1].DOF = 22;  ANOVATable.AnovaTable[1].SSq = 1312.8193776;ANOVATable.AnovaTable[1].MeanSq = 59.6736081;
			ANOVATable.AnovaTable[2].DOF = 25;  ANOVATable.AnovaTable[2].SSq = 1621.5585229;	
			ANOVATable.Fvalue = 1.72459906;
			ANOVATable.Pvalue = 0.19117273;
			ANOVADescStats DescStats[4];
			DescStats[0].N = 4; DescStats[0].Mean = 83.91189; DescStats[0].SD = 7.35196; DescStats[0].SEM = 3.67598;
			DescStats[1].N = 6; DescStats[1].Mean = 82.07600; DescStats[1].SD = 7.37151; DescStats[1].SEM = 3.00940;
			DescStats[2].N = 7; DescStats[2].Mean = 75.30680; DescStats[2].SD = 5.07543; DescStats[2].SEM = 1.91833;
			DescStats[3].N = 9; DescStats[3].Mean = 76.30106; DescStats[3].SD = 9.51584; DescStats[3].SEM = 3.17195;
			MeanCompStats  MeanCompstats[6];
			nIntervalType = ANOVA_BONFERRON;
 			nRet = stats_anova_mean_comparison(nIntervalType,DescStats, &ANOVATable, dAlpha,iLevels,MeanCompstats,6);   
		}		
	Parameters:
		nIntervalType=Input interger that indicates which method to be used
						ANOVA_BONFERRON = 1:	the Bonferroni method is used
						ANOVA_TUKEY = 2:		the Tukey-Kramer method is used
						ANOVA_SCHEFFE = 3:		the Scheffe method is used
						ANOVA_SIDAK = 4:		the Dunn-Sidak method is used
						ANOVA_FISHER = 5:		the Fisher's LSD method is used
						ANOVA_BONHOLM = 6:		the Bonferroni-Holm method is used
						ANOVA_SIDAKHOLM = 7:	the Sidak-Holm method is used
						ANOVA_DUNNETT = 8:		the Dunnett method is used
		DescStats=[input] structure containing descriptive statistics results from stats_anova_one_way
		ANOVATable=[input] reference to struct containing ANOVA table
		dAlpah=Input significant level of the model
		iLevels=Input number of groups		
		MeanCompstats=[output] pointer to structure containing statistics of difference of means of two Samples
		nSizeMeanComp=Input size (number of structs) of MeanCompstats, which should not be less than iLevel*(iLevel-1)/2
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_mean_comparison(const int nIntervalType, 
								const ANOVADescStats *DescStats, 
								const RegANOVA *ANOVATable,
								const double dAlpha,
								const int iLevels,
								MeanCompStats *MeanCompstats, uint nSizeMeanComp); 
						
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to test for equal variance of One-Way ANOVA. Function ocmath_anova_equal_variance_test is
		the main computational engine.
	Example:
		void test()
		{
			int nRet;
			int iOption;
			RegANOVA ANOVATable;
			int iLevels;	
			vector vData_1 ={77.7111,88.07729,92.15046,77.70871,72.25362,  //Group 1
							 83.52885,75.17097,87.96739,91.59955,81.93563, //Group 2
							 78.15473,79.42937,83.52293,71.4648,72.90409}; //Group 3
			vector<int> vIndex_1 = { 5,5,5 };
			iLevels = vIndex_1.GetSize();
			ANOVADescStats DescStats_1[3];
			iOption = ANOVA_LEVENE;
			nRet = stats_anova_equal_variance_test(vData_1,iLevels,vIndex_1,
												   DescStats_1,3,&ANOVATable,iOption);
		}
	Parameters:
		vData=Input vector containing data points
		iLevels=Input number of groups
		vIndex=Input vector containing the data number of each group.
		DescStats=Output reference to struct containing descriptive statistics of each group.
		nSizeDescStat=Input size (number of structs) of DescStats, which should not be less than iLevel.
		ANOVATable=Output reference to struct containing ANOVA table
		iOption=Input interger that indicates which statistic to be used.
			ANOVA_LEVENESQ = 1:		the Levene Squared stastic ()^2 is used
			ANOVA_LEVENE = 2:		the Levene stastic | | is used
			ANOVA_BROWN = 3:		the Brown=Forsythe stastic is used
			ANOVA_CHISQ = 4:		the Chi-Square stastic is used	
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_equal_variance_test(const 	vector& vData,
									int 	iLevels,
									const 	vector<int>& vIndex,
									ANOVADescStats *DescStats,uint nSizeDescStats,
									RegANOVA		*ANOVATable,
									int 	iOption);
									
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute actual or hypothetical powers for one way ANOVA. 
		The main engine of this function is ocmath_anova_powers
	Example:
		void tt1()
		{
			int nRet,nLevels;
			double dSigLevel = 0.05;
			ANOVAStats TotalStats;
		    RegANOVA ANOVATable;
			vector vPowers;
			vector vData = { 77.7111,88.07729,92.15046,77.70871,72.25362,78.62515,85.23966,70.83567,79.41474,91.14475,66.22726,90.35227,72.39297,77.58991,76.73526,57.93992,80.75954,82.81923,78.20339,80.60228, 	//Group 1
							 83.52885,75.17097,87.96739,91.59955,81.93563,89.47048,78.08122,75.17685,67.86902,86.43165,77.01454,81.61197,93.49869,85.32084,95.34041,77.72239,97.09996,81.93963,92.92086,83.46398,	//Group 2			
							 78.15473,79.42937,83.52293,71.4648,72.90409,76.00594,56.95944,86.2977,71.75579,80.70996,74.29516,77.23214,74.74264,87.19654,69.48614,69.98844,71.68327,70.78063,88.55816,81.34672 };   //Group 3		
			vector<int> vIndex = { 20 , 20 , 20 };				
			nLevels =  vIndex.GetSize();		
			ANOVADescStats DescStats[3];		
			nRet = stats_anova_one_way(vData,nLevels,vIndex,DescStats,3,TotalStats,&ANOVATable);
			vector vSampleSizes = { 20 , 30 , 37 , 60 , 80 , 100  };
			nRet = stats_anova_powers(dSigLevel, vSampleSizes,&ANOVATable, vPowers);
		}	
	Parameters:
		dSigLevel=Input significance level for power computations
		vSampleSizes=Input vector containing actual and hypothetical sample sizes
		ANOVATable=Input structure containing one way ANOVA results output from stats_anova_one_way
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_powers(const double dSigLevel, const vector& vSampleSizes, 
	 RegANOVA* ANOVATable, vector& vPowers);

	 
	 
	 
/////////////////////////////////////////////////////////////////////////////////
//
//				Independent Sample Sizes and Powers Functions									
//
/////////////////////////////////////////////////////////////////////////////////

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical powers for a one sample t-Test.
	Parameters:
		dSigLevel=Input significance level for power computations
		iTailType=Input number indicates compute powers for a two-tailed, upper tailed, or lower tailed test
		vSampleSizes=Input vector containing actual and hypothetical sample sizes
		tPowerOpts=Input structure containing power and sample size input options
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_tTest_powers1(const double dSigLevel, const int iTailType, const vector& vSampleSizes, // Use vector double (vSampleSizes) for convenience
	PowerOpts& tPowerOpts, vector& vPowers);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to computer hypothetical sample sizes for a one sample t-Test.
	Example:
		void test_samplesize()
		{
			double dSigLevel = 0.05;
			int iTailType;
			vector vSampleSizes;
			PowerOpts powerOpts;
			vector<double> vPowers;	
			powerOpts.Mean1=98.6;
 			powerOpts.Mean2=98.249;
 			powerOpts.SD=0.73318; 	
 			int nRet;
			iTailType=TAILED_TEST_TWO;	
			vSampleSizes.SetSize(3);	
			vPowers.SetSize(vSampleSizes.GetSize());
			vPowers[0]=0.2;
			vPowers[1]=0.4;
			vPowers[2]=0.6;	
			double dLowestPow;
			nRet=stats_tTest_sample_size(dSigLevel, iTailType, vPowers,powerOpts, vSampleSizes,dLowestPow); 
		}
	Parameters:
		dSigLevel=Input significance level for power computations
		iTailType=Input number indicates compute powers for a two-tailed, upper tailed, or lower tailed test
		vPowers=Input vector containing powers
		tPowerOpts=Input structure containing power and sample size input options
		vSampleSizes=Output vector containing hypothetical sample sizes
		dLowestPow=Output the lowest power, which that all powers we specified should not be less than. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_tTest_sample_size(const double dSigLevel, const int iTailType, const vector& vPowers, 
						   PowerOpts& tPowerOpts, vector& vSampleSizes, double& dLowestPow=-1);

	
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical powers for a two sample t-Test.
	Example:
		void test()
		{
			double dSigLevel;
			int iTailType;
			vector vSampleSizes;
			PowerOpts powerOpts;
			vector<double> vPowers;
			powerOpts.Mean1=2.58;
 			powerOpts.Mean2=2.85;
 			powerOpts.SD=2.83049;
 			int nRet;
			int bReturn;
 			vector<double> vChkPowers;
			dSigLevel=0.05;
			iTailType=TAILED_TEST_TWO;
			vSampleSizes.SetSize(3);
			vSampleSizes[0]=2;
			vSampleSizes[1]=4;
			vSampleSizes[2]=6;
			vPowers.SetSize(vSampleSizes.GetSize());
			nRet=stats_tTest_powers1(dSigLevel,iTailType,vSampleSizes,powerOpts, vPowers);
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vSampleSizes=Input actual and hypothetical sample sizes		
		tPowerOpts=Input structure containing power and sample size input options
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_powers1(const double dSigLevel, const HypotTestOptions& opt, const vector& vSampleSizes,
	 PowerOpts& tPowerOpts, vector& vPowers);     

	 
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical sample size for a two sample t-Test.
	Example:
		void test()
		{
			double dSigLevel;
			HypotTestOptions htopt;
			vector vSampleSizes;
			PowerOpts powerOpts;
			vector<double> vPowers;
			powerOpts.Mean1=2.58;
 			powerOpts.Mean2=2.85;
 			powerOpts.SD=2.83049;
 			int nRet;
			dSigLevel=0.05;
			htopt.HypotValue = 0;
			htopt.TailType = TAILED_TEST_TWO;
			vSampleSizes.SetSize(3);
			vPowers.SetSize(vSampleSizes.GetSize());
			vPowers[0]=0.2;
			vPowers[1]=0.4;
			vPowers[2]=0.6;
			double dLowestPow;
			nRet=stats_two_sample_tTest_sample_size(dSigLevel,htopt,vPowers,powerOpts,vSampleSizes, dLowestPow);
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vPowers=Input vector containing hypothetical powers
		tPowerOpts=Input structure containing power and sample size input options
		vSampleSizes=Output vector containing sample sizes		
		dLowestPow=Output the lowest power, which that all powers we specified should not be less than. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_sample_size(const double dSigLevel, const HypotTestOptions& opt, const vector& vPowers,
	 PowerOpts& tPowerOpts, vector& vSampleSizes, double& dLowestPow=-1);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical powers for a pair t-Test.
	Example:
		void test()
		{
			double dSigLevel;
			HypotTestOptions htopt;
			int iTailType;
			vector vSampleSizes;
			PowerOpts powerOpts;
			vector<double> vPowers;
			powerOpts.Mean1=1.23;
 			powerOpts.Mean2=3.21;
 			powerOpts.SD=2.22;
 			int nRet;
			int bReturn;
 			vector<double> vChkPowers;
			dSigLevel=0.05;
			htopt.HypotValue = 0;
			htopt.TailType = TAILED_TEST_TWO;
			vSampleSizes.SetSize(3);
			vSampleSizes[0]=2;
			vSampleSizes[1]=4;
			vSampleSizes[2]=6;
			vPowers.SetSize(vSampleSizes.GetSize());
			nRet=stats_two_sample_tTest_powers_pair1(dSigLevel, htopt, vSampleSizes, powerOpts, vPowers);  
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vSampleSizes=Input vector containing actual and hypothetical sample sizes		
		tPowerOpts=Input structure containing power and sample size input options
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_powers_pair1(const double dSigLevel, const HypotTestOptions& opt, const vector& vSampleSizes,
	                                    PowerOpts& tPowerOpts, vector& vPowers);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical Sample Size for a pair t-Test.
	Example:
		void test()
		{
			double dSigLevel;
			HypotTestOptions htopt;
			vector vSampleSizes;
			PowerOpts powerOpts;
			vector<double> vPowers;
			powerOpts.Mean1=2.58;
 			powerOpts.Mean2=2.85;
 			powerOpts.SD=2.83049;
 			int nRet;
			dSigLevel=0.05;
			htopt.HypotValue = 0;
			htopt.TailType = TAILED_TEST_TWO;
			vSampleSizes.SetSize(3);
			vPowers.SetSize(vSampleSizes.GetSize());
			vPowers[0]=0.2;
			vPowers[1]=0.4;
			vPowers[2]=0.6;
			double dLowestPow;
			nRet=stats_two_sample_tTest_sample_size_pair(dSigLevel,htopt,vPowers,powerOpts,vSampleSizes, dLowestPow);
		}	
	Parameters:
		dSigLevel=Input significance level for power computations
		opt=Input structure containing t-Test options
		vPowers=Input vector containing hypothetical powers
		tPowerOpts=Input structure containing power and sample size input options
		vSampleSizes=Output vector containing sample sizes		
		dLowestPow=Output the lowest power, which that all powers we specified should not be less than. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_two_sample_tTest_sample_size_pair(const double dSigLevel, const HypotTestOptions& opt, const vector& vPowers, 
	 PowerOpts& tPowerOpts, vector& vSampleSizes, double& dLowestPow=-1);

	 
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical powers for a one way ANOVA.
		The main computational engine of this function is ocmath_anova_powers1.
	Example:
		void test()
		{
			double dSigLevel = 0.05;
			vector vSampleSizes;
			ANOVAPowerOpts opts;
			vector vPowers;
			int nRet;
			vSampleSizes.SetSize( 3 );
			vSampleSizes[0] = 2;
			vSampleSizes[1] = 4;
			vSampleSizes[2] = 6;
			opts.Level = 5;
			opts.CSS = 4;
			opts.SD = 2.34;
			nRet = stats_anova_powers1( dSigLevel, vSampleSizes, opts, vPowers );
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		vSampleSizes=Input vector containing actual and hypothetical sample sizes
		PowerOpts=Input structure containing ANOVA power and sample size input options
		vPowers=Output vector containing powers
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_powers1(const double dSigLevel, const vector& vSampleSizes, 
	ANOVAPowerOpts& tPowerOpts, vector& vPowers);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function to compute hypothetical sample size for a one way ANOVA.
		The main computational engine of this function is ocmath_anova_sample_size.
	Example:
		void test()
		{
			double dSigLevel = 0.05;
			vector vSampleSizes;
			ANOVAPowerOpts opts;
			vector vPowers;
			int nRet;
			vPowers.SetSize( 3 );
			vPowers[0] = 0.087;
			vPowers[1] = 0.184;
			vPowers[2] = 0.293;
			opts.Level = 5;
			opts.CSS = 4;
			opts.SD = 2.34;
			nRet = stats_anova_sample_size( dSigLevel, vPowers, opts, vSampleSizes );
		}		
	Parameters:
		dSigLevel=Input significance level for power computations
		vPowers=Input vector containing powers
		PowerOpts=Input structure containing power and sample size input options
		vSampleSizes=Output vector containing hypothetical sample sizes
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/
int stats_anova_sample_size(const double dSigLevel, const vector& vPowers, 
	ANOVAPowerOpts& tPowerOpts, vector& vSampleSizes);


// Category last updated by Echo on 03/22/2007
/**$
		Function to put Powers to a structure
	Parameters:
		vPowers=Input vector containing powers
		vSampleSizes=Output vector containing hypothetical sample sizes
		dSigLevel=Input significance level for power computations
		PowerResults=Output pointer to our power output structure.
		nPowerResults=Input value of size of PowerResults, which should be the same as size of vPowers. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/	
int stats_power_output(const vector& vPowers, const vector& vSampleSize, const double dSigLevel, PowerResults* stPowers, const int nPowerResults);

// Category last updated by Echo on 03/22/2007
/**$
		Function to put Sample Sizes to structure
	Parameters:
		vSampleSizes=Output vector containing hypothetical sample sizes
		vPowers=Input vector containing powers
		dSigLevel=Input significance level for power computations
		SampleSizeRes=Output pointer to our power output structure
		nSampleSizeRes=Input value of size of SampleSizeRes, which should be the same as size of vPowers. 
	Return:
		Returns STATS_NO_ERROR on successful exit and a non-zero STATS error code on failure.
*/	
int stats_samplesize_output(const vector& vPowers, const vector& vSampleSize, const double dSigLevel, SAMPResults* stSampleSize, const int nSampleSizeRes);

	
/////////////////////////////////////////////////////////////////////////////////
//
//				Two Way ANOVA Functions									
//
/////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Two-Way ANOVA function to compute the Two-Way ANOVA F table. The Origin Two-Way ANOVA dialog
		box uses a linear regression approach in order to support unequal cell sizes (However, empty
		cells are not supported). The reference for Origin's two-way analysis of variance is Applied
		Linear Statistical Models, Neter J., Kutner M., Nachtsheim C., and Wasserman W. (1996), The
		McGraw-Hill Companies, Inc., Boston, MA. See Section 22.2 for a discussion of how to use dummy
		variables for testing both factor main effects and interaction effects. The Origin Two-Way ANOVA
		dialog box also makes use of the LLVC functions ocmath_dummy_vars to create the necessary
		design matrices and ocmath_multiple_linear_regression to perform the linear regressions of the
		design matrices. The results of the linear regressions are then used to construct the two-way
		ANOVA table.
	Example:
		void test_anova_two_way()
		{
			int nLevelsA, nLevelsB, iInteractions, nPts, nRet;				
			vector vDEP_VAR={ 236,280,201,215,239,187,203,196,199,216,185,177,
							  204,180,211,235,160,187,213,181,182,145,163,148 };					
			vector<int> vFACTOR_A = {1,1,1,1,1,1,1,1,1,1,1,1,
									 2,2,2,2,2,2,2,2,2,2,2,2 };									 
			vector<int> vFACTOR_B = {1,1,1,1,2,2,2,2,3,3,3,3,
									 1,1,1,1,2,2,2,2,3,3,3,3 };			 
			nLevelsA = 2; nLevelsB = 3; 		
			ANOVA2DescStats ADescStats[2], BDescStats[3],  ToDescStats, ABDescStats[6];			
			ANOVA2Table arANOVA2_Table[6];			
			iInteractions = 1;			
			nRet = stats_anova_two_way_compute_anova_table(iInteractions,
														   vFACTOR_A,nLevelsA,
														   vFACTOR_B,nLevelsB,
														   vDEP_VAR,
														   &ADescStats,&BDescStats,
														   ToDescStats,&ABDescStats,
														   arANOVA2_Table);
		}
	Parameters:
		iInteractions=Flag indicating whether (1) or not (0) interaction effects are to be determined
		vFACTOR_A=Vector containing integerized level indices (1 based) for Factor A
		nLevelsA=The number of Factor A levels
		vFACTOR_B=Vector containing integerized level indices (1 based) for Factor B
		nLevelsB=The number of Factor D levels
		vDEP_VAR=Vector containing dependent variable data values
		ADescStats=Output structure pointer to descriptive statistics results of FactorA
		BDescStats=Output structure pointer descriptive statistics results of FactorB
		ToDescStats=Output structure containing descriptive statistics results of whole data.
		ABDescStats=Output structure containing descriptive statistics results of FactorA*FactorB		
		arANOVA2_Table=A five(if no interaction) or six(if interaction) element array of an RegANOVA structure 
					   holding all computed F Table values.
					   one element for each row in the ANOVA F table (A, B, (A * B), Model, Error and Corrected Total).
	Return:
		Returns a five(if no interaction) or six(if interaction) element array of an RegANOVA structure 
		holding all computed F Table values which has one element for each row in the ANOVA F table 
		(A, B, (A * B), Model, Error and Corrected Total) and returns element array of an ANOVA2DescStats structure 
		holding descriptive statistics for each group of A, group of B and group of A*B. 
		Also returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
int stats_anova_two_way_compute_anova_table(const int iInteractions, const vector<int> &vFACTOR_A, const int nLevelsA, const vector<int> &vFACTOR_B,
	 const int nLevelsB, const vector<double> &vDEP_VAR, ANOVA2DescStats *ADescStats, ANOVA2DescStats *BDescStats, ANOVA2DescStats& ToDescStats, 
	 ANOVA2DescStats *ABDescStats, ANOVA2Table* arANOVA2_Table);


// Category last updated by Joseph on 09/11/2007
/** >Statistics>Advanced
		Two-Way ANOVA function to perform a Means Comparison analysis for a single factor of the
		Two-Way ANOVA dialog box.
	Example:
		void test_anova_two_way_means_comparison()
		{
			int iTestType, nLevelsA, nLevelsB, iInteractions, nPts, nRet;	
			double dSigLevel = 0.05;	
			vector vDEP_VAR={ 236,280,201,215,239,187,203,196,199,216,185,177,
							  204,180,211,235,160,187,213,181,182,145,163,148 };		
			nPts = vDEP_VAR.GetSize();	
			vector<int> vFACTOR_A = {1,1,1,1,1,1,1,1,1,1,1,1,
									 2,2,2,2,2,2,2,2,2,2,2,2 };	                         
			vector<int> vFACTOR_B = {1,1,1,1,2,2,2,2,3,3,3,3,
									 1,1,1,1,2,2,2,2,3,3,3,3 };	 
			nLevelsA = 2; nLevelsB = 3; 	
			ANOVA2DescStats ADescStats[2], BDescStats[3],  ToDescStats, ABDescStats[6];	
			ANOVA2Table arANOVA2_Table[6];	
			MeanCompStats ANOVA2_MeanCompStatsA[1];
			MeanCompStats ANOVA2_MeanCompStatsB[3];
			iInteractions = 1;	
			iTestType = ANOVA_BONFERRON;	
			nRet = stats_anova_two_way_compute_anova_table(iInteractions,
														   vFACTOR_A,nLevelsA,
														   vFACTOR_B,nLevelsB,
														   vDEP_VAR,
														   &ADescStats,&BDescStats,
														   ToDescStats,&ABDescStats,
														   arANOVA2_Table);
														   
			if (STATS_NO_ERROR == nRet )
			{
				nRet = stats_anova_two_way_means_comparison(iTestType, vFACTOR_A, 
															nLevelsA, vDEP_VAR, 
															arANOVA2_Table, dSigLevel, 
	 														ANOVA2_MeanCompStatsA,1);
			}
			if (STATS_NO_ERROR == nRet )
			{
				nRet = stats_anova_two_way_means_comparison(iTestType, vFACTOR_B, 
															nLevelsB, vDEP_VAR, 
															arANOVA2_Table, dSigLevel, 
	 														ANOVA2_MeanCompStatsB,3);
			}
		}
	Parameters:
		iTestType=Inout bitwise integer specifying which Means Comparison test(s) if any to run:
						ANOVA_BONFERRON = 1:	the Bonferroni method is used
						ANOVA_TUKEY = 2:		the Tukey-Kramer method is used
						ANOVA_SCHEFFE = 3:		the Scheffe method is used
						ANOVA_SIDAK = 4:		the Dunn-Sidak method is used
						ANOVA_FISHER = 5:		the Fisher's LSD method is used
						ANOVA_BONHOLM = 6:		the Bonferroni-Holm method is used
						ANOVA_SIDAKHOLM = 7:	the Sidak-Holm method is used
						ANOVA_DUNNETT = 8:		the Dunnett method is used
		vFactor=[input] vector containing Factor 'A' or Factor 'B'
		nLevels=Input number of levels
		vDEP_VAR=[input] vector containing the dependent variable data 
				 whose levels are indicated by the paired vector vFACTOR
		arANOVA2_Table=[input] reference to struct containing ANOVA table
		dSigLevel=Input significance Level of the model
		ANOVA2_MeanCompStats=[output] structure containing mean comparison statistics of 2-way ANOVA. 
		nSizeMeanComp=Input size(number of structure) of ANOVA2_MeanCompStats, it should not be less than nLevels*(nLevels-1)/2
	Return:
		Computes Two-Way ANOVA Means Comparison results. Also returns
		STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
int stats_anova_two_way_means_comparison(const int iTestType, const vector<int> &vFactor, 
										 const int nLevels,	const vector &vDEP_VAR,
										 const ANOVA2Table *arANOVA2_Table, const double dSigLevel, 
	 									 MeanCompStats *ANOVA2_MeanCompStats, uint nSizeMeanComp);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Two-Way ANOVA function used to compute and output Two-Way ANOVA Power Analysis.
	Example:
		See function stats_anova_two_way above for a sample call.
	Parameter:
		iInteractions=Input enable (1) or disable (0) computation of Two-Way ANOVA interactions
		dSigLevel=Alpha used to compute Power
		arANOVA2_Table=A four element array of an RegANOVA structure holding all computed F Table
			values having one element for each row of the ANOVA F table (A, B, A * B, and Error)
		vSampleSizes=Input vector of actual or hypothetical sample sizes
		vPowers= Output vector of all the power computed, vSSPOW[0] is the actual power
		iSource=Input flag to decide whose power to be computed
					0 - Factor A Power
					1 - Factor B Power
					2 - Factor A*B Power		
	Return:
		Outputs the Two-Way ANOVA Power Analysis table to the Results Log. Returns STATS_NO_ERROR 
		on successful exit or an STATS error code on failure.
*/

int stats_anova_two_way_power(const int iInteractions, const double dSigLevel,
	  const ANOVA2Table *arANOVA2_Table,const vector &vSampleSizes, vector& vPowers,
	  const int iSource = 0);



/////////////////////////////////////////////////////////////////////////////////
//
//				Survival Analysis Functions									
//
/////////////////////////////////////////////////////////////////////////////////
	 
// Category last updated by Joseph on 09/11/2007
/** >Statistics>Advanced
		Survival Analysis function used to compute the Kaplan-Meier Estimate of survival 
		probalities for a sample of failure times.
	Example:
		void test_kaplan()
		{
			vector vTime_1 = { 107,90,169,50,954,972,6,111,70,90,930,52,50,414,111,482,104,29,983,93,133,468,65,846 };
			vector vCensor_1 = { 0,1,0,1,0,0,1,1,0,2,0,0,0,2,0,1,0,1,1,0,2,0,0,1 };
			vector pCensored = {0, 2};
			int nCensored = pCensored.GetSize();
			double dConLevel = 0.90;
				
			SAEstim sSurvivEstim[24];
			SAQuantEst sSAQuantEstim[3];
			SASummary stSASumStats;

			int nRet = stats_kaplan_meier_surviv_fn(vTime_1, vCensor_1, pCensored, dConLevel,  
												sSurvivEstim, 24, sSAQuantEstim, 3,  stSASumStats);
		}
	Parameter:
		vTime=Input vector containing time values
		vCensor=Input vector containing censor values
		vCensored=Input vector constaing Censored value. Please make sure that value of pCensored are 
		all in vCensor and value are unduplicated. Size of vCensored needs to be less than 20.
		psSurvivEstim=Output structure containing the Survival Analysis of time
		nSizeSAEstm=Input size (number of structure) of psSurvivEstim. it should not be less than size of vTime
		psSAQuantEstim=Output structure containing Quartile estimates
		nSizeQuanEstm=Input size (number of structure) of psSAQuantEstim. It should not be less than 3.
		stSASumStats=Output structure containing summary of event and censored values 
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
///Echo 7/15/04 change the prototype so that censord value could be multi-selected. 
int	stats_kaplan_meier_surviv_fn(const vector& vTime, const vector& vCensor,const vector& vCensored, const double dConLevel,
	SAEstim *psSurvivEstim, uint nSizeSAEstm, SAQuantEst *psSAQuantEstim, uint nSizeQuanEstm, SASummary& stSASumStats);	 
//int	stats_kaplan_meier_surviv_fn(const vector& vTime, const vector& vCensor, const double dCensored, const double dConLevel,
//	SAEstim *psSurvivEstim, SAQuantEst *psSAQuantEstim, SASummary& stSASumStats);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Survival Analysis function used to compute parameter estimates and other statistics associated with 
		the Cox Proportional Hazards Model for fixed covariates.
	Example:
		void test()
		{
			int nRet,nCensored,ii;
			uint nSizeParaEstm,nSizeSAEstm;
			SASummary	stSASummary;
			SAFitStats	stSAFitStats;
			vector vTime_1a = { 107,90,169,50,954,972,6,111,70,90,930,52,50,414,111,482,104,29,983,93,133,468,65,846 };
			vector vCensor_1 = { 0,1,0,1,0,0,1,1,0,0,0,0,0,0,0,1,0,1,1,0,0,0,0,1 };
			vector vCensored_1a = { 0 };
			matrix mCovars_1 = {{1,	100},
				{1,	100},
				{0,	300},
				{0,	100},
				{1,	300},
				{0,	300},
				{1,	200},
				{0,	300},
				{1,	200},
				{0,	200},
				{1,	100},
				{1,	200},
				{0,	300},
				{0,	100},
				{1,	300},
				{0,	100},
				{1,	200},
				{0,	200},
				{0,	100},
				{1,	300},
				{1,	300},
				{0,	200},
				{0,	100},
				{0,	200} };
			SAParaEstim	sSAParaEstim_1[2];
			nSizeParaEstm = 2;
			SAEstim 	sSurvivEstim_1[24];
			nSizeSAEstm = 24;
			nRet =  stats_surviv_cox_model(vTime_1a,vCensor_1,vCensored_1a,mCovars_1,stSASummary,stSAFitStats,sSAParaEstim_1,
					nSizeParaEstm,sSurvivEstim_1,nSizeSAEstm);
		}		
	Parameter:
		vTime=Input vector containing time values
		vCensor=Input vector containing censor values
		vCensored=Input vector constaing Censored value. Please make sure that value of pCensored are 
		all in vCensor and value are unduplicated. Size of vCensored needs to be less than 20.
		mCovars=Input matrix containing predictor variables (covariates)
		stSASummary=Output structure containing summary of event and censored values 
		psSAFitStats=Output structure containing the statistics assessing the overall fit of model
		psSAParaEstim=Output structure containing parameter estimates 
		nSizeParaEstm=Input size (number of structure) of psSAParaEstim. It should not be less than the number of covariates
		psSurvivEstim= Output structure containing the Survival Analysis of time
		nSizeSAEstm=Input size (number of structure) of psSurvivEstim. it should not be less than size of vTime		
		mCov=Output variance-covariance matrix of the parameter estimates. 
		mCorr=Output correlation matrix of the parameter estimates. 
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	 
///Echo 7/15/04 change the prototype so that censord value could be multi-selected. 
int	stats_surviv_cox_model(const vector& vTime, const vector& vCensor, const vector& vCensored, const matrix& mCovars,
	SASummary& stSASummary, SAFitStats &stSAFitStats, SAParaEstim *psSAParaEstim, uint nSizeParaEstm, SAEstim *psSurvivEstim, uint nSizeSAEstm, matrix& mCov = NULL, matrix& mCorr = NULL);
//int	stats_surviv_cox_model(const vector& vTime, const vector& vCensor, const double dConsored, const matrix& mCovars,
	//SASummary& stSASummary, SAFitStats &stSAFitStats, SAParaEstim *psSAParaEstim, SAEstim *psSurvivEstim);
	
	
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Survival Analysis function used to compute the parameter estimates and other 
		statistics that are associated with the Weibull model. The main engine is 
		imsls_d_survival_glm.
	Example:
	Parameter:
		vTime=Input vector containing time values
		vCensor=Input vector containing censor values
		vCensored=Input vector constaing Censored value. Please make sure that value of pCensored are 
		all in vCensor and value are unduplicated. Size of vCensored needs to be less than 20.Data points should not be all censored. 
		mCovars=Input matrix containing predictor variables (covariates)
		dCriterion=Optimized criterion.The criterion to be maximized is a constant plas the log-likehood. 
		stSASummary=Output structure containing summary of event and censored values. 
		psSAParaEstim=Output structure containing parameter estimates. 
		nSizeParaEstm=Output size (number of structure) of psSAParaEstim.
		psSurvivEstim= Output structure containing the Survival Analysis of time
		nSizeSAEstm=Input size (number of structure) of psSurvivEstim. it should not be less than size of vTime				
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	 
int	stats_surviv_weibull(const vector& vTime, const vector& vCensor,const vector& vCensored, const matrix& mClass, const matrix& mContinuous, 
	double& dCriterion, SASummary& stSASummary, SAParaEstim *psSAParaEstim, int& nSizeParaEstm, SAEstim *psSurvivEstim, uint nSizeSAEstm);

	
// Category last updated by Iris on 03/14/2007
/**$
		List out the censored value in vector vCensor
	Parameter:
		vCensor=Input vector containing censor values
		vCensList=Output vector containing alll unduplicated censor values. Size of vCensList should be less than 20 
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/			
int stats_censored_value_list(const vector& vCensor, vector& vCensList);
	
/////////////////////////////////////////////////////////////////////////////////
//
//				Non-Parameter Hypothesis Analysis								
//
/////////////////////////////////////////////////////////////////////////////////	
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs the Wilcoxon signed rank test on a single sample. To test a particular sample 
		came from a popular with a specified median.
	Example:
	Parameter:
		vData=Input vector of data points for Wilcoxon Sign Rank test
		dMedian=Input median test value
		nTailType=Input value indicates Tail Type. which containing 3 types:
					TAILED_TEST_TWO = 0,
					TAILED_TEST_UPPER =1,
					TAILED_TEST_LOWER = 2
		Wilco1Summary=Output structure containing summary statistics of the data
		Wilco1Stats=Output structure containing the test statistic and p-value
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/
int stats_one_sample_wilcoxon(const vector& vData, const double dMedian,const int nTailType, NPHSummary& Wilco1Summary, NPHStats& Wilco1Stats);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs a Wilcoxon rank sum test for identical population 
		distribution functions.. The main engine is imsls_d_wilcoxon_rank_sum
	Example:
	Parameter:
		vData1=Input vector containing 1st data points for Wilcoxon Rank Sum test
		vData2=Input vector containing 2nd data points for Wilcoxon Rank Sum test
		nTailType=Input value indicates Tail Type. which containing 3 types:
					TAILED_TEST_TWO = 0,
					TAILED_TEST_UPPER =1,
					TAILED_TEST_LOWER = 2
		dWilcoxonW=Output values of Wilcoxon static, W.
		WilcoSummary=Output structure containing summary statistics of the 2 groups
						WilcoSummary[0]: Summary statistics of 1st group
						WilcoSummary[1]: Summary statistics of 2nd group						
		WilcoRanks=Output structure containing Statistics of Ranks
					WilcoRanks[0]: Statistics of Negative Ranks
					WilcoRanks[1]: Statistics of Positive Ranks					
		WilcoStats=Output structure containing the test statistic and p-value, No corrected for ties.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure. If ties exists, STATS_WARNING_AT_LEAST_ONE_TIE
		will be returned in addition to the results. 
*/	 
int stats_wilcoxon_rank_sum(const vector& vData1, const vector& vData2, const int nTailType, double &dWilcoxonW, NPHSummary *WilcoSummary, 
	NPHRanks *WilcoRanks, NPHStats &WilcoStats);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs a Mann-Whitney U test for on two independent samples of possibley unequal 
		size. The main engine is ocmath_mann_whitney
	Example:
	Parameter:
		vData1=Input vector containing 1st data points for Wilcoxon Rank Sum test
		vData2=Input vector containing 2nd data points for Wilcoxon Rank Sum test
		nTailType=Input value indicates Tail Type. which containing 3 types:
					TAILED_TEST_TWO = 0,
					TAILED_TEST_UPPER =1,
					TAILED_TEST_LOWER = 2
		MannSummary=Output structure containing summary statistics of the 2 groups
						MannSummary[0]: Summary statistics of 1st group
						MannSummary[1]: Summary statistics of 2nd group						
		MannRanks=Output structure containing Statistics of Ranks
					MannRanks[0]: Ranks of 1st group
					MannRanks[1]: Ranks of 2nd group					
		MannStats=Output structure containing the test statistic and p-value, No corrected for ties.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	 
int stats_mann_whitney(const vector& vData1, const vector& vData2, const int nTailType, NPHSummary *MannSummary, 
	NPHRanks *MannRanks, NPHStats &MannStats, const bool bExact = true);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs a Wilcoxon signed rank test for paired samples. The main engine 
		is imsls_d_wilcoxon_sign_rank
	Example:
	Parameter:
		vData1=Input vector containing 1st data points for Wilcoxon Rank Sum test
		vData2=Input vector containing 2nd data points for Wilcoxon Rank Sum test
		nTailType=Input value indicates Tail Type. which containing 3 types:
					TAILED_TEST_TWO = 0,
					TAILED_TEST_UPPER =1,
					TAILED_TEST_LOWER = 2
		WilSiSummary=Output structure containing summary statistics of the 2 groups
						WilcoSummary[0]: Summary statistics of 1st group
						WilcoSummary[1]: Summary statistics of 2nd group						
		WilSiRanks=Output structure containing Statistics of Ranks
					WilcoRanks[0]: Statistics of Positive Ranks
					WilcoRanks[1]: Statistics of Negative Ranks					
		WilSiStats=Output structure containing the test statistic and p-value. No corrected for ties.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	 
int stats_wilcoxon_sign_rank(const vector& vData1, const vector& vData2, const int nTailType, NPHSummary *WilSiSummary, 
	NPHRanks *WilSiRanks, NPHStats &WilSiStats);


// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs a Kruskal-Wallis test for identical population medians. The main engine 
		is imsls_d_kruskal_wallis_test
	Example:
	Parameter:
		vData=Input vector containing data points
		iLevels=Input number of groups
		vIndex=Input vector containing the data number of each group.
		KruSummary=Output structure containing summary statistics of each groups
						KruSummary[i-1]: Summary statistics of i-th group
		KruRanks=Output structure containing Statistics of Ranks
					KruRanks[i-1]: Ranks of i-th group
		KruStats=Output structure containing the test statistic and p-value, No corrected for ties.
		KruAdjStats=Output structure containing the test statistic and p-value, Adjusted for ties.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	
int stats_kruskal_wallis_anova(const vector& vData, const int iLevels, const vector& vIndex,
	NPHSummary *KruSummary, NPHRanks *KruRanks, NPHStats &KruStats);

// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Performs a Friedman two-way analysis of variance for a randomized complete block design.
	Example:
	Parameter:
		vDEP_VAR=Vector containing dependent variable data values
		vBlocks=Vector containing integerized blocks
		vTreaments=Vector containing integerized treaments
		FrieSummary=Output structure containing summary statistics of each groups
						FrieSummary[i-1]: Summary statistics of i-th group
		FrieRanks=Output structure containing Statistics of Ranks
					FrieRanks[i-1]: Ranks of i-th group
		FrieStats=Output structure containing the test statistic and p-value, No corrected for ties.
	Return:
		Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.
*/	
int stats_friedman_anova(const vector& vDEP_VAR, const vector vBlocks, const vector& vTreaments,
	NPHSummary *FrieSummary, NPHRanks *FrieRanks, NPHStats &FrieStats);

	

/////////////////////////////////////////////////////////////////////////////////
//
//				ROC Curve									
//
/////////////////////////////////////////////////////////////////////////////////
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Function for ROC curve, finds the hit rates in dataset over a range of thresholds.
		Hits are "true positives" in one dataset and "false positives" in the other and the
		plot of true positive probality(TPP) versus false positive probability(FPP) is called
		the receive operating charateristic(ROC) curve. A ROC curve is a visual tool for examining
		tradeoffs between the sensitivity and selectivity of a classifier. 
	Example:
		#define ROC_MAX_THRESHOLD	1000
		void stats_roc_curve_example()
		{
			vector vData = { 107,90,169,50,954,972,6,111 };
			vector vState = { 0,1,2,1,2,0,1,1 };
			vector vStateList = { 0,2 };
			
			int nThreshold;
			ocmath_threshold_list(vData, vData.GetSize(), &nThreshold, false);
			
			ROCArea stROCArea;
			ROCCuv pROC[ROC_MAX_THRESHOLD];
			
			int nRet = stats_roc_curve(vData, vState, vStateList, stROCArea, pROC, nThreshold, 0.95, ROC_INTERPOLATION);
			//stROCArea.Area = 0.9375;
			//stROCArea.SE = 0.089759;
			//stROCArea.Prob = 0.0433;
			//stROCArea.LCI = 0.761578;
			//stROCArea.UCI = 1.11342;
			
			vector vThreshold = {50, 100, 150, 200, 250};	
			nRet = stats_roc_curve(vData, vState,  vStateList, stROCArea, pROC, vThreshold.GetSize(), 0.95, ROC_USER_DEFINED, &vThreshold);
			//stROCArea.Area = 0.9583;
			//stROCArea.SE = 0.34485;
			//stROCArea.Prob = 0.034264;
			//stROCArea.LCI = 0.28243;
			//stROCArea.UCI = 1.63423;
		}			
	Parameter:
		vData=Input vector of data points. 
		vState=Input vector of state variable 
		vStaValList=Input vector of Value list state variable.
		sROCArea = Outputstructure of related variables of Area under ROC curve
		psROC = Output pointer to structure to ROC curve coordinate points.
		nROC=Input size of ROC curve coordinate. 
		dConfLevel=Input confidence level for confidence band of area under the curve. 
		nThdFlag=Input flag for Threshold List defination. 
			nThdFlag=ROC_INTERPOLATION,  Thresholds are interpolation of data points
			nThdFlag=ROC_SPEED_MODE, Thredholds are bounds for 100 equally spaced intervals
			nThdFlag=ROC_USER_DEFINED, Using user defined threshold list
		vThres=Optional input vector of user defined threshold list. Usable only if nThdFlag=ROC_USER_DEFINED
*/
int stats_roc_curve(const vector vData, const vector vState, const vector vStaValList, ROCArea& sROCArea, ROCCuv *psROC, uint nROC, double dConfLevel = 0.95, int nThdFlag = 0, const vector* pvThres = NULL);
	
///Echo 1/5/05 ADD_GROUP_DESC_STAT
// Category last updated by Echo on 03/22/2007
/** >Statistics>Advanced
		Compute descriptive statistics for each group
*/
void desc_stats_group(const vector& vData, const vector& vGroup, DescStatResults *pDescStats, QuantileResults *pQuantiles, int nOverall = 0);

///Echo 5/10/06 MOVE_TO_ANALYSIS_C
///move ananlysis related functions to analysis.c
//bool trim_nanum_1group(vector &vData, vector<string> &vstrGroup, int& nMissing);
//
//bool trim_nanum_2group(vector &vData, vector<string> &vsGroup1, vector<string> &vsGroup2, int& nMissing);
///END MOVE_TO_ANALYSIS_C

// Category last updated by Echo on 03/22/2007
/**$
*/
void InitQuantiles(QuantileOptions& opQuantiles);

// Category last updated by Echo on 03/22/2007
/**$
*/
void InitDescStats(DescStatOptions& opDescStats);

///Echo 5/10/06 MOVE_TO_ANALYSIS_C
///move ananlysis related functions to analysis.c
//int  trim_nanum_sa(int* nMissing, vector &vTime, vector &vCensor, vector<string> &vGroup=NULL, matrix &mCovariat=NULL);

///Echo 5/10/06 MOVE_TO_REPORT_UTILS_C
//void report_tree_create(ReportTree& outRT, LPCSTR lpsczLabel);

///Echo 5/10/06 MOVE_TO_ANALYSIS_C
///move ananlysis related functions to analysis.c

///*
		//To get 2 dataset from grouped range
	//Parameter:
		//rInGroup=[input]Input data range. 
		//vData1=[output]1st dataset got from data range
		//vData2=[output]2nd dataset got from data range
		//vstrLab=[output]vector containing Label of the 2 dataset
		//nMissing=[output]Number of missing values in range
	//Return:
		//Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.	
//*/
//int get_2_data_in_group(const Range& rInGroup, vector& vData1, vector& vData2, vector<string>& vstrLab, int& nMissing);
//
///*
		//To get 2 dataset from data range
	//Parameter:
		//rInVar=[input]Input data range. 
		//vData1=[output]1st dataset got from data range
		//vData2=[output]2nd dataset got from data range
		//vstrLab=[output]vector containing Label of the 2 dataset
		//nMissing=[output]Number of missing values in range
		//bPair=[input] bit to control how to trim missing value. If bPair = true, missing values will be trimmed pairwise, otherwise will be trimmed one by one. 
	//Return:
		//Returns STATS_NO_ERROR on successful exit or an STATS error code on failure.	
//*/
//int get_2_data_in_var(const Range& rInVar, vector& vData1, vector& vData2, vector<string>& vstrLab, int& nMissing, const bool bPair = false);
///end MOVE_TO_ANALYSIS_C


///Echo 5/10/06 MOVE_TO_REPORT_UTILS_C
///move report related functions to report_utils.c

//void rt_add_nph_descriptive(ReportTree& outRT, const DescStatResults *pDescStats, const QuantileResults *pQuantiles, const vector<string>& vstrGroup, const int nGroups);
////void nph_descriptive(ReportTree& outRT, DescStatResults *pDescStats, QuantileResults *pQuantiles, vector<string> vStrGroup, int nGroups);
//
//void rt_add_sa_summary(ReportTree& outRT, SASummary stSASumStats, LPCSTR lpsczLab, const int nIndex = 0);
////void sa_summary(ReportTree& outRT, SASummary stSASumStats, LPCSTR lpsczLab);
//
//void  report_tree_make_result_picture(ReportTable& rt, ReportData& rd, int nIndex, int imageIndex = 0);
//
//BOOL  calculate_results_curve_update_report_data(ReportTree& outRT, ReportData& outData, ReportTable& rtFitCurves,  const TreeNode& plot, double conf, vector& vx, vector& vy, vector vErrors, int nPoints, int nIndex, int nPlot = 0, bool bOneLayer = TRUE);
//
//void  update_plot_curve(ReportTree& outRT, ReportData& outData, const TreeNode& plot, const int& nGroup = 1, const int& nPlot = 0, bool bOneLayer = TRUE);
//
//void create_plot_curve_table(ReportTree& outRT, const TreeNode& plot, const int& nGroup = 1, bool bOneLayer = TRUE);
///end MOVE_TO_REPORT_UTILS_C

#endif // _STATS_UTILS_H